/*  
	BeOS Front-end du PDF file reader xpdf.
    Copyright (C) 1998 Hubert Figuiere

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
//
// PDFView.cpp
//
// (c) 1998-99 Hubert Figuiere
//
// $Id: PDFView.cpp,v 1.7 1999/03/27 00:50:01 hub Exp $
//


#include <stdio.h>
#include <storage/Path.h>
#include <interface/ScrollBar.h>
#include <interface/PrintJob.h>

#include "PDFWindow.h"
#include "PDFView.h"
#include "PDFExceptions.h"


// zoom factor is 1.2 (similar to DVI magsteps)
static const int kZoomDPI[MAX_ZOOM - MIN_ZOOM + 1] = {
	29, 35, 42, 50, 60,
	72,
	86, 104, 124, 149, 179
};

#define DEF_ZOOM	0


inline static float RealSize (float x, float zoomDPI)
{
	return zoomDPI / 72 * x;
}


///////////////////////////////////////////////////////////////////////////
PDFView::PDFView (entry_ref * ref, BRect frame,
								const char *name,
								uint32 resizeMask,
								uint32 flags)
	: BView (frame, name, resizeMask, flags)
{
	mCurrentPage = 0;
	mDoc = NULL;
	mOk = false;
	mRotation = 0.0f;
	mZoom = DEF_ZOOM;
	mBitmap = NULL;
	mPrintSettings = NULL;

	GString * title = NULL;
	GString * fileName = NULL;
	BEntry entry (ref, true);
	BPath path;
	
	entry.GetPath (&path);
	
	fileName = new GString (path.Path ());	// putting path.Leaf() makes BeOS freeze :-(
	mDoc = new PDFDoc (fileName);
	if (! mDoc->isOk ()) {
		delete mDoc;
		mDoc = NULL;
		throw (DocException ("Doc not created"));
		return;
	}
	mTitle = new GString("BePDF: ");
	mTitle->append (path.Leaf ());
	
	mWidth = RealSize (mDoc->getPageWidth(1), kZoomDPI[mZoom - MIN_ZOOM]);
	mHeight = RealSize (mDoc->getPageHeight(1), kZoomDPI[mZoom - MIN_ZOOM]);

	RecreateBitmap ();

	mOffscreenView = new BView(mBitmap->Bounds(), "", B_FOLLOW_NONE, 0);
	mBitmap->AddChild (mOffscreenView);
	mOutputDev = new BeOutputDev (mOffscreenView);
	mOk = true;
}


///////////////////////////////////////////////////////////////////////////
PDFView::~PDFView()
{
	delete mBitmap;
	delete mDoc;
	delete mOutputDev;
	delete mTitle;
}



///////////////////////////////////////////////////////////////////////////
void
PDFView::Draw(BRect updateRect)
{
	if (mBitmap == NULL) {
#ifdef DEBUG
		fprintf (stderr, "WARNING: PDFView::Draw() NULL bitmap\n");
#endif
		return;
	}
	DrawBitmap (mBitmap, updateRect, updateRect);
}


///////////////////////////////////////////////////////////////////////////
void
PDFView::FrameResized (float width, float height)
{
	FixScrollbars();
}


///////////////////////////////////////////////////////////////////////////
void
PDFView::AttachedToWindow ()
{
	Window()->SetTitle (mTitle->getCString());
}

///////////////////////////////////////////////////////////////////////////
void
PDFView::KeyDown (const char * bytes, int32 numBytes)
{
	switch (*bytes) {
	case B_PAGE_UP:
	case B_BACKSPACE:
		SetPage (mCurrentPage - 1);
		break;
	case B_PAGE_DOWN:
	case B_SPACE:
		SetPage (mCurrentPage + 1);
		break;
	case B_HOME:
		SetPage (1);
		break;
	case B_END:
		SetPage (GetNumPages ());
		break;
	default:
		BView::KeyDown (bytes, numBytes);
		break;
	}
}


///////////////////////////////////////////////////////////////////////////
void
PDFView::FixScrollbars ()
{
	BRect frame = Bounds();
	BScrollBar * scroll;
	float x, y;
	float bigStep, smallStep;

	x = mWidth - frame.right;
	if (x < 0.0) {
		x = 0.0;
	}
	y = mHeight - frame.bottom;
	if (y < 0.0) {
		y = 0.0;
	}
	
	scroll = ScrollBar (B_HORIZONTAL);
	scroll->SetRange (0.0, x);
	scroll->SetProportion ((mWidth - x) / mWidth);
	bigStep = frame.right - 2;
	smallStep = bigStep / 10.;
	scroll->SetSteps (smallStep, bigStep);

	scroll = ScrollBar (B_VERTICAL);
	scroll->SetRange (0.0, y);
	scroll->SetProportion ((mHeight - y) / mHeight);
	bigStep = frame.bottom - 2;
	smallStep = bigStep / 10.;
	scroll->SetSteps (smallStep, bigStep);
}


///////////////////////////////////////////////////////////////////////////
void 
PDFView::Redraw()
{
	PDFWindow * parentWin;
	BRegion *clip;
	BRect aRect;

	if (!mOutputDev) {
		return;
	}
	
	mWidth = RealSize (mDoc->getPageWidth (mCurrentPage), kZoomDPI[mZoom - MIN_ZOOM]);
	mHeight = RealSize (mDoc->getPageHeight (mCurrentPage), kZoomDPI[mZoom - MIN_ZOOM]);

	parentWin = dynamic_cast<PDFWindow *> (Window());
	if (parentWin != NULL) {
		BMessage * notify = new BMessage (PDFWindow::PAGE_CHANGE_NOTIFY_MSG);
		
		notify->AddInt32 (PDFWindow::PAGE_MSG_LABEL, mCurrentPage);
		parentWin->PostMessage (notify);
		delete notify;
		
		parentWin->SetZoomSize (mWidth, mHeight);
		parentWin->PostMessage (notify);
	}

	mBitmap->Lock ();
	mOffscreenView->RemoveSelf ();
	mBitmap->Unlock ();
	
	RecreateBitmap ();
	mOffscreenView->ResizeTo (mWidth, mHeight);
	mBitmap->AddChild (mOffscreenView);

	mBitmap->Lock ();
	rgb_color hColor = mOffscreenView->HighColor ();
	mOffscreenView->SetHighColor (255, 255, 255);
	mOffscreenView->FillRect (Frame ());
	mOffscreenView->SetHighColor (hColor);
	FillRect (Bounds(), B_SOLID_LOW);
	mDoc->displayPage (mOutputDev, mCurrentPage, kZoomDPI[mZoom - MIN_ZOOM], mRotation, gTrue);
	mOffscreenView->Sync ();
	mBitmap->Unlock ();
	DrawBitmap (mBitmap, BPoint (0, 0));
}


///////////////////////////////////////////////////////////////////////////
void 
PDFView::SetPage (int page)
{
	printf ("PDFView::SetPage (%d)\n", page);
	if (mCurrentPage != page) {
		if (page < 1) {
			mCurrentPage = 1;
		}
		else if (page > GetNumPages()) {
			mCurrentPage = GetNumPages();
		}
		else {
			mCurrentPage = page;
		}
		Redraw ();
		FixScrollbars ();
	}
}


//////////////////////////////////////////////////////////////////
void
PDFView::SetZoom (int zoom)
{
	if (mZoom != zoom) {
		mZoom = zoom;
		Redraw ();
		FixScrollbars ();
	}
}




//////////////////////////////////////////////////////////////////
void
PDFView::RecreateBitmap ()
{
	delete mBitmap;
	mBitmap = new BBitmap(BRect (0, 0, mWidth, mHeight), 
							B_RGB_32_BIT, true, false);
}

#ifdef __MWERKS__
#pragma mark ------printing-------
#endif

///////////////////////////////////////////////////////////////////////////
/*
	based up Be sample code in "Be Newsletter Volume 2, Issue 18 -- May 6, 1998"
*/

status_t
PDFView::PageSetup()
{
	status_t  result = B_ERROR;

	BPrintJob  printJob(this->mTitle->getCString());


	if (mPrintSettings != NULL) {
		/* page setup has already been run */
		printJob.SetSettings(new BMessage(mPrintSettings));
	}

	result = printJob.ConfigPage();

	if (result == B_NO_ERROR) {
	
		delete (mPrintSettings);
		mPrintSettings = printJob.Settings();
	}

	return result;
}




///////////////////////////////////////////////////////////////////////////
void 
PDFView::Print()
{
	BPrintJob printJob(this->mTitle->getCString());

	/* save the page number because printing will mess it up */
	int savedPage = Page ();

	if (mPrintSettings == NULL) {
		if (PageSetup() != B_NO_ERROR) {
			return;
		}
	}

	printJob.SetSettings(new BMessage(mPrintSettings));

	if (printJob.ConfigJob() == B_NO_ERROR) {
		int32  curPage = 1;
		int32  firstPage;
		int32  lastPage;
		int32  pagesInDocument;
		BRect  pageRect = printJob.PrintableRect();
		BRect  curPageRect = pageRect;

		pagesInDocument = mDoc->getNumPages ();
		firstPage = printJob.FirstPage();
		lastPage = printJob.LastPage();

		if (lastPage > pagesInDocument) {
			lastPage = pagesInDocument;
		}

		printJob.BeginJob();

		for (curPage = firstPage; curPage <= lastPage; curPage++) {
			SetPage (curPage);
			curPageRect = mBitmap->Bounds();

			printJob.DrawView(this, curPageRect, BPoint(0.0, 0.0));

			printJob.SpoolPage();

			if (!printJob.CanContinue()) {
				goto catastrophic_exit;
			}
		}
		printJob.CommitJob();
	}

catastrophic_exit:;

	/* restore the page */
	SetPage (savedPage);
}

