/*  
	BeOS Front-end of PDF file reader xpdf.
    Copyright (C) 1997 Benoit Triquet
    Copyright (C) 1998-99 Hubert Figuiere

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	$Id: BeOutputDev.cpp,v 1.14 1999/06/09 21:34:21 hub Exp $

*/
//
// Be OutputDev : device for BeOS output for Xpdf
//

#include <stdlib.h>

#include <interface/Shape.h>
#include <interface/View.h>
#include <interface/Region.h>
#include <interface/Polygon.h>
#include <interface/Bitmap.h>

#include <gtypes.h>
#include <GfxState.h>
#include <GfxFont.h>

#include "BeOutputDev.h"



static const rgb_color WHITE_COLOR = {255,255,255};
static const rgb_color LINK_COLOR = { 0, 0, 255 };


/////////////////////////////////////////////////////////////////////////
BeOutputDev::BeOutputDev(BView * view)
	: mView (view)
{
	mState = NULL;
	mClip = NULL;
	mKludgedEncoding = B_ISO_8859_1;
}


/////////////////////////////////////////////////////////////////////////
BeOutputDev::~BeOutputDev()
{
	_State *s;
	while (mState != NULL) {
		s = mState;
		mState = s->next;
		delete s;
	}
	delete mClip;
}


#pragma mark -


/////////////////////////////////////////////////////////////////////////
void BeOutputDev::startPage(int /*pageNum*/, GfxState * /*state*/)
{
	_State * s;
	// erase view
	mView->SetHighColor(WHITE_COLOR);
	mView->FillRect(mView->Bounds());

	while (mState != NULL) {
		s = mState;
		mState = s->next;
		delete s;
	}
	mState = NULL;
	delete mClip;
	mClip = new BRegion ();
	mClip->Set (mView->Bounds());
	mView->ConstrainClippingRegion (mClip);
}



/////////////////////////////////////////////////////////////////////////
void BeOutputDev::dump()
{
	mView->Flush();
}



/////////////////////////////////////////////////////////////////////////
void BeOutputDev::drawLinkBorder( double x1, double y1, double x2,
 double y2, double w)
{
	int x, y;
	BPoint pt1, pt2, pt3, pt4;

#ifdef DEBUG
	fprintf (stderr, "xpdf: BeOutputDev::drawLinkBorder() called\n");
	fprintf (stderr, "xpdf: w = %lf\n", w);
#endif

	mView->SetPenSize (w);
	mView->BeginLineArray (4);
	cvtUserToDev (x1, y1, &x, &y);
	pt1.x = x;
	pt1.y = y;
	cvtUserToDev (x2, y1, &x, &y);
	pt2.x = x;
	pt2.y = y;
	cvtUserToDev (x2, y2, &x, &y);
	pt3.x = x;
	pt3.y = y;
	cvtUserToDev (x1, y2, &x, &y);
	pt4.x = x;
	pt4.y = y;
	mView->AddLine (pt1, pt2, LINK_COLOR);
	mView->AddLine (pt2, pt3, LINK_COLOR);
	mView->AddLine (pt3, pt4, LINK_COLOR);
	mView->AddLine (pt4, pt1, LINK_COLOR);
	mView->EndLineArray ();
}


/////////////////////////////////////////////////////////////////////////
void BeOutputDev::saveState( GfxState * /*state*/ )
{
	_State *st = new _State;
	st->next = mState;
	st->low = mView->LowColor( );
	st->high = mView->HighColor( );
	st->clip.MakeEmpty ();
	st->clip.Include (mClip);
	mState = st;
}



/////////////////////////////////////////////////////////////////////////
void BeOutputDev::restoreState( GfxState * /*state*/ )
{
	if( mState == NULL ) {
		return;
	}
	
	_State *st = mState;
	mView->SetLowColor (st->low);
	mView->SetHighColor (st->high);
	mClip->MakeEmpty ();
	mClip->Include (&st->clip);
	mView->ConstrainClippingRegion (mClip);
	mState = st->next;
	delete st;
}


/////////////////////////////////////////////////////////////////////////
void BeOutputDev::updateAll( GfxState *state )
{
	double *ctm = state->getCTM( );
	updateCTM( state, ctm[ 0 ], ctm[ 1 ], ctm[ 2 ], ctm[ 3 ], ctm[ 4 ],
				ctm[ 5 ] );
	updateLineDash( state );
	updateFlatness( state );
	updateLineJoin( state );
	updateLineCap( state );
	updateMiterLimit( state );
	updateLineWidth( state );
	updateFillColor( state );
	updateStrokeColor( state );
	updateFont( state );
}



/////////////////////////////////////////////////////////////////////////
void BeOutputDev::updateFlatness( GfxState *state )
{
#ifdef DEBUG
	fprintf (stderr, "xpdf: updateFlatness()\n");
#endif

	mFlatness = state->getFlatness ();
}



void BeOutputDev::updateLineWidth(GfxState *state)
{
	mView->SetPenSize (state->getLineWidth());
}


/////////////////////////////////////////////////////////////////////////
void BeOutputDev::updateFillColor( GfxState *state )
{

	GfxColor *co = state->getFillColor ();
	mFill.red = 0xFF * co->getR ();
	mFill.green = 0xFF * co->getG ();
	mFill.blue = 0xFF * co->getB ();
}


/////////////////////////////////////////////////////////////////////////
void BeOutputDev::updateStrokeColor( GfxState *state )
{
	GfxColor *co = state->getStrokeColor ();
	mStroke.red = 0xFF * co->getR ();
	mStroke.green = 0xFF * co->getG ();
	mStroke.blue = 0xFF * co->getB ();
}


/////////////////////////////////////////////////////////////////////////
static struct { char *name, *family, *style; } _fonts[ ] = {
	{ "Helvetica", "Swis721 BT", "Roman" },
	{ "Helvetica-Oblique", "Swis721 BT", "Italic" },
	{ "Helvetica-Bold", "Swis721 BT", "Bold" },
	{ "Helvetica-BoldOblique", "Swis721 BT", "Bold Italic" },
	{ "Times-Roman", "Dutch801 Rm BT", "Roman" },
	{ "Times-Italic", "Dutch801 Rm BT", "Italic" },
	{ "Times-Bold", "Dutch801 Rm BT", "Bold" },
	{ "Times-BoldItalic", "Dutch801 Rm BT", "Bold Italic" },
	{ "Courier", "Courier10 BT", "Roman" },
	{ "Courier-Oblique", "Courier10 BT", "Italic" },
	{ "Courier-Bold", "Courier10 BT", "Bold" },
	{ "Courier-BoldOblique", "Courier10 BT", "Bold Italic" },
	{ "Palatino-Roman", "Baskerville", "Roman" },
	{ "Palatino-Bold", "Baskerville", "Bold" },
	{ "Palatino-Italic", "Baskerville", "Italic" },
	{ "Palatino-BoldItalic", "Baskerville", "Bold Italic" },
	{ "Symbol", "SymbolProp BT", "Regular" },
	{ "ZapfDingbats", "Dingbats", "Regular" },
	0
};

// index: {symbolic:12, fixed:8, serif:4, sans-serif:0} + bold*2 + italic

void BeOutputDev::updateFont (GfxState *state)
{
#ifdef MORE_DEBUG
	fprintf (stderr, "xpdf: BeOutputDev::updateFont() called\n");
#endif

	char *name, *family, *style;
	GfxFont *gfx_font = state->getFont ();
	GString *fontName;
	char * plus;
	
	if( gfx_font == NULL ) {
		return;
	}
	fontName = gfx_font->getName();
	if (fontName == NULL) {
#ifdef MORE_DEBUG
		fprintf (stderr, "fontName == NULL\n");
#endif
		return;
	}
	name = fontName->getCString();
	
	/*
	   huge kludge : built-in font names begins with some code followed
	   by a '+'. Find this '+' and remove it 
	 */
	plus = strchr (name, '+');
	if (plus != NULL) {
		plus++;
		/* it is a build-in font. Let's unmangle the name */
		memmove (name, plus, strlen (plus) + 1);
	}
	
#ifdef MORE_DEBUG
	fprintf (stderr, "xpdf: Requested font is %s\n", name);
#endif
	int i = 0;
	while (_fonts[i].name && (strcmp(name, _fonts[i].name) != 0)) {
		i ++;
	}
#ifdef MORE_DEBUG
	fprintf (stderr, "Found font index is %d\n", i);
#endif
	if (_fonts [i].name) {
		family = _fonts [i].family;
		style = _fonts [i].style;
	} 
	else {
		int i = 0;
		if (gfx_font->isFixedWidth ()) {
			i = 8;
		}
		else if (gfx_font->isSerif ()) {
			i = 4;
		}
		if (gfx_font->isBold ()) {
			i += 2;
		}
		if (gfx_font->isItalic ()) {
			i += 1;
		}
		family = _fonts [i].family;
		style = _fonts [i].style;
	}
	
#ifdef MORE_DEBUG
	fprintf (stderr, "Font type is %d\n", gfx_font->getType());
	fprintf (stderr, "Embed font name %s\n", gfx_font->getEmbeddedFontName());
#endif
	
	mFont.SetFamilyAndStyle (family, style);
	mFont.SetSize (state->getTransformedFontSize());
	mFont.SetEncoding (mKludgedEncoding);
	mFont.SetSpacing (B_STRING_SPACING);
	mView->SetFont (&mFont);
}


void BeOutputDev::ForceEncodingKludge (uint8 encoding)
{
	mKludgedEncoding = encoding;
}

/////////////////////////////////////////////////////////////////////////
void BeOutputDev::drawChar(GfxState *state, double x, double y,
			double /*dx*/, double /*dy*/, Guchar c)
{
	double x1, y1;
	
	if ((state->getRender() & 3) == 3) {
		return; /* for Acrobat Capture: (invisible text) */
	}
	
	state->transform( x, y, &x1, &y1 );
	if (state->getRender() & 1) {
		mView->SetHighColor(mStroke);
	}
	else {
		mView->SetHighColor(mFill);
	}
	mView->DrawChar( c, BPoint( x1, y1 ) );
	mView->SetHighColor(mStroke);
}


/////////////////////////////////////////////////////////////////////////
/*
void BeOutputDev::drawString (GfxState *state, GString *s)
{
	double x1, y1;
	state->transform( state->getCurX( ), state->getCurY( ), &x1, &y1);
	mView->DrawString (s->getCString(), BPoint (x1, y1));
}
*/

/////////////////////////////////////////////////////////////////////////
GBool BeOutputDev::upsideDown()
{
#ifdef DEBUG
	fprintf (stderr, "xpdf: BeOutputDev::upsideDown() called\n");
#endif

	return gTrue;
}


/////////////////////////////////////////////////////////////////////////
GBool BeOutputDev::useDrawChar()
{
#ifdef DEBUG
//	fprintf (stderr, "xpdf: BeOutputDev::useDrawChar() called\n");
#endif

	return gTrue;
}


/////////////////////////////////////////////////////////////////////////
void BeOutputDev::drawImage(GfxState *state, Stream *str, int width,
			 int height, GfxImageColorMap *colorMap,
			 GBool inlineImg)
{
#ifdef IMAGE_DEBUG
	fprintf (stderr, "BeOutputDev::drawImage() called\n");
	fprintf (stderr, "width = %d; height = %d\n", width, height);
	fprintf (stderr, "inline ? %d\n", inlineImg);
#endif

	double xt, yt, wt, ht;
	BBitmap * bitmap = NULL;
	int nBits, nVals, nComps;
	Guchar pixel [4] = { 0, 0, 0, 0 };
	uint8 * pixLine = NULL;
	GfxColor currentColor;
	bool rotate, xFlip, yFlip;	//iamge rotation parameters.
	int j, k;
	int dk;					// increment step for image
	float x0, y0;			// top left corner of image
	float w0, h0;		// size of image
	int32 bpr;			//bytes per row
	
	state->transform (0, 0, &xt, &yt);
	state->transformDelta (1, 1, &wt, &ht);
	if (wt > 0) {
		x0 = xt;
		w0 = wt;
	}
	else {
		x0 = xt + wt;
		w0 = -wt;
	}
	if (ht > 0) {
		y0 = yt;
		h0 = ht;
	} 
	else {
		y0 = yt + ht;
		h0 = -ht;
	}
	/*
		compute rotation parameters.
	*/	
	state->transformDelta (1, 0, &xt, &yt);
	rotate = fabs (xt) < fabs (yt);
	if (rotate) {
		xFlip = ht < 0;
		yFlip = wt > 0;
	}
	else {
		xFlip = wt < 0;
		yFlip = ht > 0;
	}

	
	nComps = colorMap->getNumPixelComps();
	nVals = width * nComps;
	nBits = colorMap->getBits();
	
	bitmap = new BBitmap (BRect (0, 0, width -1, height - 1), B_RGB32);

	bpr = bitmap->BytesPerRow ();
	pixLine = (uint8*) malloc (bpr);

	// initialize the image stream
	str->resetImage(width, nComps, nBits);

	if (yFlip) {
		dk = -1;
	}
	else {
		dk = 1;
	}

	for (k = (yFlip?height - 1:0); (yFlip?k >= 0:k < height) ; k += dk ) {
		if (xFlip) {
			/* flip horizontally the pixels */
			j = (width - 1) * 3;	
			while (j >= 0) {
				str->getImagePixel (pixel);
				colorMap->getColor (pixel, &currentColor);
				pixLine [j] = 255.f * currentColor.getR();
				pixLine [j + 1] = 255.f * currentColor.getG();
				pixLine [j + 2] = 255.f * currentColor.getB();
				j -= 3;
			}
		}
		else {
			/* get pixels in the normal order */
			j = 0;	
			while (j < (width * 3)) {
				str->getImagePixel (pixel);
				colorMap->getColor (pixel, &currentColor);
				pixLine [j] = 255.f * currentColor.getR();
				pixLine [j + 1] = 255.f * currentColor.getG();
				pixLine [j + 2] = 255.f * currentColor.getB();
				j += 3;
			}
		}
		bitmap->SetBits (pixLine, width * 3, (k * bpr), B_RGB32);
	}

	mView->DrawBitmap (bitmap, BRect (x0, y0, x0 + w0, y0 + h0));
	
	delete bitmap;
}



/////////////////////////////////////////////////////////////////////////
void BeOutputDev::drawImageMask(GfxState *state, Stream *str,
			     int width, int height, GBool invert,
			     GBool inlineImg)
{
#ifdef DEBUG
	fprintf (stderr, "xpdf: BeOutputDev::drawImageMask() called\n");
#endif

}



/////////////////////////////////////////////////////////////////////////
/*
	Stroke the current path stored in state
*/
void BeOutputDev::stroke(GfxState *state)
{
	int i;
	GfxPath *path = state->getPath();
	GfxSubpath *subPath;
	int numSubPath;
	rgb_color oldColor;

	oldColor = mView->HighColor ();
	mView->SetHighColor (mStroke);
	
	numSubPath = path->getNumSubpaths();
	for (i = 0; i < numSubPath; i++) {	
		subPath = path->getSubpath(i);

		BShape * shape = new BShape ();
		SubPathToShape (state, subPath, shape);
		mView->MovePenTo (B_ORIGIN);
		mView->StrokeShape (shape);
		delete shape;
	}
	mView->SetHighColor (oldColor);
}



/////////////////////////////////////////////////////////////////////////
/*
	Fill the current path stored in state
	BShape seems really buggy in R4 PowerPC (untested on Intel, but tech 
	support to not have it: they run Intel of course :-( )
*/
void 
BeOutputDev::fill(GfxState *state)
{
	int i;
	GfxPath *path = state->getPath();
	GfxSubpath *subPath;
	int numSubPath;
	rgb_color oldColor;
	BShape * shape;
	
	oldColor = mView->HighColor ();
	mView->SetHighColor (mFill);
	numSubPath = path->getNumSubpaths();
	for (i = 0; i < numSubPath; i++) {	
		subPath = path->getSubpath(i);

		shape = new BShape ();
		SubPathToShape (state, subPath, shape);
		mView->MovePenTo (B_ORIGIN);
		mView->FillShape (shape);
		delete shape;

	}
	mView->SetHighColor (oldColor);
}


void 
BeOutputDev::eoFill(GfxState *state)
{
	fill (state);
}

/////////////////////////////////////////////////////////////////////////
void
BeOutputDev::clip(GfxState *state)
{
#ifdef DEBUG
	fprintf (stderr, "xpdf: BeOutputDev::clip()\n");
#endif

/*	BRegion * region;*/
	BRegion * currentClip, * newClip;
	int i;
	GfxPath *path = state->getPath();
	GfxSubpath *subPath;
	int numSubPath;
	BShape *shape;
	BPicture * pict;
	BView * offView;

	pict = new BPicture ();
	offView = new BView (mView->Frame(), "", B_FOLLOW_NONE, 0);
	mView->AddChild (offView);
	offView->BeginPicture (pict);
	numSubPath = path->getNumSubpaths();
	for (i = 0; i < numSubPath; i++) {	
		subPath = path->getSubpath(i);

		shape = new BShape ();
		SubPathToShape (state, subPath, shape);
		offView->MovePenTo (B_ORIGIN);
		offView->FillShape (shape);
		delete shape;
	}

	offView->EndPicture ();	
	mView->RemoveChild (offView);
	currentClip = new BRegion ();
	mView->ClipToPicture (pict, B_ORIGIN, true);
	mView->GetClippingRegion (currentClip);
	mClip->IntersectWith (currentClip);
	mView->ConstrainClippingRegion (mClip);
	delete currentClip;
	
	delete pict;
}


void
BeOutputDev::eoClip(GfxState *state)
{
	clip (state);
}


#pragma mark -

/////////////////////////////////////////////////////////////////////////
/*
	Convert a GfxSubpath to a BPolygon... for Be rendering
	
	This function is currently obsolete since we use Bezier.
*/
void
BeOutputDev::AddSubPathToPoly (GfxState *state, /*const */GfxSubpath * subPath, 
                            BPolygon * poly)
/* subPath should be const, but it is not */
{
	int numPoint;
	BPoint currentPt;
	double x, y;
	int j;
	
	if (poly == NULL) {
		fprintf (stderr, "WARNING: AddSubPathToPoly(): poly NULL!\n");
	}
	numPoint = subPath->getNumPoints();
	for (j = 0; j < numPoint; j++) {
		/*
			Skip curve point. Will change when we'll do bezier curves.
		*/
		if (subPath->getCurve(j) == gFalse) {
			state->transform (subPath->getX(j), subPath->getY(j), &x, &y);
			currentPt.x = x;
			currentPt.y = y;
			poly->AddPoints (&currentPt, 1);
		}
	}
}

/////////////////////////////////////////////////////////////////////////
/*
	Stroke a GfxSubpath. Since R3, BeOS supports Bezier drawing, so I don't
	need to implement it as a polygon... and thus I make smooth curves.

	We use StrokeBezier rather than BShape() because BShape seems to close the
	path and thus does NOT draw things correctly.
*/
void
BeOutputDev::SubPathToStrokeBezier (GfxState *state, /*const */GfxSubpath * subPath, 
                            BView * view)
/* subPath should be const, but it is not */
{
	int numPoint;
	BPoint bezier[4];
	double x0, y0, x1, y1, x2, y2, x3, y3;
	int j;

	state->transform (0, 0, &x0, &y0);

#ifdef DEBUG
	if (subPath == NULL) {
		fprintf (stderr, "BeOutputDev::SubPathToStrokeBezier() subPath == NULL\n");
	}
#endif
	numPoint = subPath->getNumPoints();

	j = 0;
	while (j < numPoint) {
		if ((j >= 0) && (subPath->getCurve (j))) {
			state->transform (subPath->getX(j), subPath->getY(j), &x1, &y1);
			state->transform (subPath->getX(j + 1), subPath->getY(j + 1), &x2, &y2);
			state->transform (subPath->getX(j + 2), subPath->getY(j + 2), &x3, &y3);
			
			bezier [0] = BPoint (x0, y0);
			bezier [1] = BPoint (x1, y1);
			bezier [2] = BPoint (x2, y2);
			bezier [3] = BPoint (x3, y3);
			
			view->StrokeBezier (bezier);
			
			j +=3;
			x0 = x3;
			y0 = y3;
		}
		else {
			state->transform (subPath->getX(j), subPath->getY(j), &x1, &y1);
			if (j > 0) {
				view->StrokeLine (BPoint (x0, y0), BPoint (x1, y1));
			}
			x0 = x1;
			y0 = y1;
			j++;
		}
	}
	
}


/////////////////////////////////////////////////////////////////////////
/*
	Convert a GfxSubpath to a BShape... for Be rendering
*/
void
BeOutputDev::SubPathToShape (GfxState *state, /*const */GfxSubpath * subPath, 
                            BShape * shape)
/* subPath should be const, but it is not */
{
	int numPoint;
	BPoint bezier[3];
	double x, y;
	int j;

	shape->Clear ();
	numPoint = subPath->getNumPoints();
	
	state->transform (subPath->getX(0), subPath->getY(0), &x, &y);	
	bezier [0].x = x;
	bezier [0].y = y;
	shape->MoveTo (bezier [0]);
	for (j = 1; j < numPoint; j++) {
		/*
			Skip curve point. Will change when we'll do bezier curves.
		*/
		state->transform (subPath->getX(j), subPath->getY(j), &x, &y);
		bezier [0].x = x;
		bezier [0].y = y;
		if (subPath->getCurve (j) == gFalse) {
			shape->LineTo (bezier [0]);	
		}
		else {
			j++;
			if (j >= numPoint) {
				/* error */
#ifdef DEBUG
				fprintf (stderr, "could not complete bezier\n");
#endif
				return;
			}
			if (subPath->getCurve (j) == gFalse) {
#ifdef DEBUG
				fprintf (stderr, "not a curve point\n");
#endif
			}
			state->transform (subPath->getX(j), subPath->getY(j), &x, &y);
			bezier [1].x = x;
			bezier [1].y = y;
			j++;
			if (j >= numPoint) {
				/* error */
#ifdef DEBUG
				fprintf (stderr, "could not complete bezier\n");
#endif
				return;
			}
			if (subPath->getCurve (j) == gTrue) {
#ifdef DEBUG
				fprintf (stderr, "is a curve point\n");
#endif
			}
			state->transform (subPath->getX(j), subPath->getY(j), &x, &y);
			bezier [2].x = x;
			bezier [2].y = y;
			
			shape->BezierTo (bezier);
		}
	}
}


