#include "KTextView.h"

#include <malloc.h>
#include <string.h>
#include <Beep.h>
#include <MenuItem.h>
#include <Path.h>
#include <Bitmap.h>
#include <TranslationUtils.h>
#include <FindDirectory.h>
#include <NodeInfo.h>
#include <Alert.h>
#include <PopUpMenu.h>
#include <stdio.h>
#include <String.h>
#include <MessageQueue.h>
#include <fs_attr.h>

#include "KWindow.h"
#include "KMenuBar.h"
#include "KApp.h"
#include "KPrefs.h"
#include "KShortcut.h"

#include "KTextUtils.h"
#include "KMenuUtils.h"
#include "KTextRunUtils.h"
#include "KString.h"
#include "KSearch.h"


KTextView::KTextView(BRect frame, const char *name, BRect textRect
					, uint32 resizeMask, uint32 frags)
				:BTextView(frame, name, textRect, resizeMask, frags)
					, fUseHTMLSyntax(false)
					, fBackgroundBitmap(NULL)
					, fUndoStatus(K_UNDO_UNAVAILABLE)
					, fUndoBuffer(NULL)
					, fUndoLastDeletedTime(0)
{
	this->Initialize();
}


KTextView::~KTextView()
{
	delete		fBaseFont;
	delete		fShortcut;
	delete		fBackgroundBitmap;
	delete[]	fUndoBuffer;
}


void
KTextView::Initialize()
{
	//Set Vars.
	fBaseFont = new BFont(be_plain_font);
	fMultiButton = K_SHOW_POPUP;
	fShortcut = new KShortcut(this);
	
	
	//textview prefs.
	BFont*	font = new BFont(be_plain_font);
	KPrefs*	prefs = ((KApp*)be_app)->GetPrefs();
	const char*	family;
	const char*	style;
	float	size, tabwidth;
	bool	autoindent, wordwrap;
	int32	multibutton;
	rgb_color	aBGColor, aCharColor, aTagColor, aCommentColor;
	
	prefs->Lock();
	prefs->GetData("font_family", &family);
	prefs->GetData("font_style", &style);
	prefs->GetData("font_size", &size);
	prefs->GetData("tab_width", &tabwidth);
	prefs->GetData("auto_indent", &autoindent);
	prefs->GetData("word_wrap", &wordwrap);
	prefs->GetData("multi_button", &multibutton);
	prefs->GetData("background_color", &aBGColor);
	prefs->GetData("char_color", &aCharColor);
	prefs->GetData("tag_color", &aTagColor);
	prefs->GetData("comment_color", &aCommentColor);
	prefs->Unlock();
	
	font->SetFamilyAndStyle(family, style);
	font->SetSize(size);
	
	fCharColor = aCharColor;
	fTagColor = aTagColor;
	fCommentColor = aCommentColor;
	fBackgroundColor = aBGColor;
	
	this->SetBaseFont(font);
	this->SetTabWidth(tabwidth);
	this->SetWordWrap(wordwrap);
	this->SetAutoindent(autoindent);
	this->SetMultiButtonFunction(multibutton);
	
	this->SetStylable(true);
	this->SetDoesUndo(true);
	
	//Undo function
	fUndoBuffer = new char[1];
	*fUndoBuffer = (char)NULL;
}


void
KTextView::Cut(BClipboard* clipboard)
{
	int32	start, end;
	this->GetSelection(&start, &end);
	if(start != end)
		this->BTextView::Cut(clipboard);
}


void
KTextView::Copy(BClipboard* clipboard)
{
	int32	start, end;
	this->GetSelection(&start, &end);
	if(start != end)
		this->BTextView::Copy(clipboard);
}


void
KTextView::AttachedToWindow()
{
	this->SetViewColor(fBackgroundColor);
	
	BBitmap*	aBitmap;
	BPath		aPath;
	::find_directory(B_USER_SETTINGS_DIRECTORY, &aPath, true);
	aPath.Append(kSettingDir);
	aPath.Append(kBackgroundFolder);
	aPath.Append(kBackgroundImageName);
	
	aBitmap = BTranslationUtils::GetBitmap(aPath.Path(), NULL);
	
	if(aBitmap != NULL){
		if(aBitmap->ColorSpace() == B_RGB32){//only for 32-bit color graphics.
			int32	aLength = aBitmap->BitsLength();
			uchar*	aData = (uchar*)aBitmap->Bits();
			for(int32 i = 0; i < aLength; i++){
				*(aData + i) += (255 - *(aData + i))/2;
			}
		}
		fBackgroundBitmap = aBitmap;
		this->SetViewBitmap(fBackgroundBitmap);
	}
	
}


void
KTextView::InsertText(const char* inText, int32 inLength
							, int32 inOffset, const text_run_array* inRuns)
{	
	if(!this->IsEditable()){ ::beep(); return; }
	
	((KWindow*)this->Window())->SetDirty(true);
	
	if(system_time() - fUndoLastDeletedTime > 50000){
		//Undo function
		delete[]	fUndoBuffer;
		fUndoBuffer = new char[this->TextLength() + 1];
		strcpy(fUndoBuffer, this->Text());
		fUndoStatus = K_UNDO_INSERT;
	}
	
	
	this->BTextView::InsertText(inText, inLength, inOffset, NULL);
	
	BMessage*	aCurMessage = this->Window()->CurrentMessage();
	
	if(fUseHTMLSyntax){
		//aCurMessage->PrintToStream();
		//::printf("\n");
		/*  
		be:opcode ???
		be:string -> currently selected string.
		be:clause_start -> begining of selection.
		be:clause_end -> end of selection.
		be:selection -> array of 2 values.???
		be:confirmed -> confirming is done.
		*/
		if(aCurMessage != NULL && aCurMessage->what == 'IMEV'){
			/* IM is ACTIVE */
			bool		IsConfirmed;
			status_t	err1;
			err1 = aCurMessage->FindBool("be:confirmed", &IsConfirmed);
			if(err1 == B_NO_ERROR && IsConfirmed){
				/* Translation CONFIRMED */
				BMessage*	aMessage = new BMessage(K_TEXT_MODIFIED);
				aMessage->AddInt32("from_offset", inOffset - 1);// SHIFT one character
				aMessage->AddInt32("to_offset", inOffset + inLength - 1);
				this->Window()->PostMessage(aMessage, this);
			}
		}else{
			/* IM is INACTIVE*/
			BMessage*	aMessage = new BMessage(K_TEXT_MODIFIED);
			aMessage->AddInt32("from_offset", inOffset);
			aMessage->AddInt32("to_offset", inOffset + inLength);
			this->Window()->PostMessage(aMessage, this);
		}
	}
	
}


void
KTextView::DeleteText(int32 fromOffset, int32 toOffset)
{
	if(!this->IsEditable()){ ::beep(); return; }
	
	//KTextUtils	utils;
	
	((KWindow*)this->Window())->SetDirty(true);
	
	//Undo function
	delete[]	fUndoBuffer;
	fUndoBuffer = new char[this->TextLength() + 1];
	strcpy(fUndoBuffer, this->Text());
	fUndoStatus = K_UNDO_DELETE;
	fUndoLastDeletedTime = system_time();
	
	this->BTextView::DeleteText(fromOffset, toOffset);
	
	
	BMessage*	aCurMessage = this->Window()->CurrentMessage();
	
	if(fUseHTMLSyntax && !(aCurMessage != NULL && aCurMessage->what == 'IMEV')){
		BMessage*	aMessage = new BMessage(K_TEXT_MODIFIED);
		aMessage->AddInt32("from_offset", fromOffset);
		aMessage->AddInt32("to_offset", fromOffset);
		this->Window()->PostMessage(aMessage, this);
	}
}


void
KTextView::Undo(BClipboard* clipboard)
{
	switch(fUndoStatus)
	{
		case K_UNDO_UNAVAILABLE:
			//do nothing...
			break;
		case K_UNDO_INSERT:
		case K_UNDO_DELETE:
			{
				const char*	aText = this->Text();
				
				int32	from;
				int32	curTo = this->TextLength();
				int32	prevTo = strlen(fUndoBuffer);
				
				if(strcmp(aText, fUndoBuffer) == 0){ return; }
		
				for(from = 0; *(aText + from) == *(fUndoBuffer + from); from++)
					;
					
				for(;;){
					if(curTo == from || prevTo == from){
						break;
					}else if(curTo >= 1 && prevTo >= 1
							&& *(aText + curTo - 1) == *(fUndoBuffer + prevTo - 1)){
						curTo--;
						prevTo--;
					}else{
						break;
					}
				}
	
				char*	ntext = new char[strlen(fUndoBuffer) + 1];
				strcpy(ntext, fUndoBuffer);
				
				char*	buText = new char[strlen(aText) + 1];
				strcpy(buText, aText);
				
				this->Delete(from, curTo);
				this->Insert(from, ntext + from, prevTo - from);
				this->Select(from, prevTo);
				this->ScrollToSelection();
				
				delete[]	fUndoBuffer;
				fUndoBuffer = buText;
		
				delete[]	ntext;
			}
			break;
		case K_UNDO_DELETE_AND_INSERT:
			
			break;
		default:
			;
	}
	
	//this->BTextView::Undo(clipboard);
}


void
KTextView::MessageReceived(BMessage* msg)
{	
	//read preference...
	KPrefs*	prefs = ((KApp*)be_app)->GetPrefs();
	const char*	aPrefix;
	int32	aLineWidth;
	
	prefs->GetData("text_prefix", &aPrefix);
	prefs->GetData("line_width", &aLineWidth);
	
	key_info	aInfo;
	::get_key_info(&aInfo);
	int32		aTextLen = this->TextLength();
	
	//the current selection and the current lines...
	int32	aFrom, aTo, aLineFrom, aLineTo;
	this->GetSelection(&aFrom, &aTo);
	aLineFrom = 0;
	aLineTo = aTextLen;
	
	for(int32 i = aFrom; i >= 0; i--){
		if(*(this->Text() + i) == LF){
			aLineFrom = i + 1;
			break;
		}
	}
	if(aTo > 0 && *(this->Text() + aTo - 1) == LF){ aTo--; }
	for(int32 i = aTo; i < aTextLen; i++){
		if(*(this->Text() + i) == LF){
			aLineTo = i;
			break;
		}
	}
	if(aLineFrom > aLineTo){ aLineFrom = aLineTo; }
	
	
	//message switching...
	switch(msg->what)
	{
			
		case K_TEXT_SHIFT_PREFIX_RIGHT:
			{
				char*	aText = new char[aLineTo - aLineFrom + 1];
				this->GetText(aLineFrom, aLineTo - aLineFrom, aText);
				KString	aPrefixStr, aResString;
				aPrefixStr << "\n" << aPrefix;
				aResString << aPrefix << aText;
				aResString.ReplaceAll("\n", aPrefixStr.Text());
				
				this->Select(aLineFrom, aLineFrom);
				this->Delete(aLineFrom, aLineTo);
				
				this->Insert(aLineFrom, aResString.Text(), aResString.Length());
				this->Select(aLineFrom, aLineFrom + aResString.Length());
				
				delete[]	aText;
			}
			break;
			
		case K_TEXT_SHIFT_PREFIX_LEFT:
			{
				
				char*	aText = new char[aLineTo - aLineFrom + 1];
				this->GetText(aLineFrom, aLineTo - aLineFrom, aText);
				
				KString	aPrefixStr;
				aPrefixStr << "\n" << aPrefix;
				KString	aResString = aText;
				aResString.ReplaceAll(aPrefixStr.Text(), "\n");
				
				if(aResString.StartsWith(aPrefix)){
					aResString.Delete(0, 1);
				}
				
				this->Select(aLineFrom, aLineFrom);
				this->Delete(aLineFrom, aLineTo);
				
				this->Insert(aLineFrom, aResString.Text(), aResString.Length());
				
				this->Select(aLineFrom, aLineFrom + aResString.Length());
				
				delete[]	aText;
			}
			break;
		
		case K_TEXT_SHIFT_RIGHT:
			{
				char*	aText = new char[aLineTo - aLineFrom + 1];
				this->GetText(aLineFrom, aLineTo - aLineFrom, aText);
				KString	aResString;
				aResString << "\t" << aText;
				aResString.ReplaceAll("\n", "\n\t");
				
				this->Select(aLineFrom, aLineFrom);
				this->Delete(aLineFrom, aLineTo);
				
				this->Insert(aLineFrom, aResString.Text(), aResString.Length());
				
				this->Select(aLineFrom, aLineFrom + aResString.Length());
				delete[]	aText;
			}
			break;
			
		case K_TEXT_SHIFT_LEFT:
			{
				char*	aText = new char[aLineTo - aLineFrom + 1];
				this->GetText(aLineFrom, aLineTo - aLineFrom, aText);
				
				KString	aResString = aText;
				aResString.ReplaceAll("\n\t", "\n");
				if(aResString.StartsWith("\t")){
					aResString.Delete(0, 1);
				}
				
				this->Select(aLineFrom, aLineFrom);
				this->Delete(aLineFrom, aLineTo);
				
				this->Insert(aLineFrom, aResString.Text(), aResString.Length());
				
				this->Select(aLineFrom, aLineFrom + aResString.Length());
				delete[]	aText;
			}
			break;
		
		case K_TEXT_WRAP_LINE:
			{
				//Buggy, buggy buggy!!!!!!!!!
				//KTextUtils	aUtils;
				char*		aText = new char[aLineTo - aLineFrom + 1];
				
				this->GetText(aLineFrom, aLineTo - aLineFrom, aText);
				
				KString		aResString, aLine, aCurrentLine, aChar;
				KString		aTextString = aText;
				int32		aCharNum, aCurrentWidth, aLineNum = aTextString.CountParagraphs();
				
				for(int32 i = 0; i < aLineNum; i++){
					aLine = aTextString.Paragraph(i);
					aCurrentLine = "";
					aCurrentWidth = 0;
					aCharNum = aLine.CountCharacters();
					for(int32 j = 0; j < aCharNum; j++){
						aChar = aLine.Character(j);
						aCurrentWidth++;
						if(aChar.Length() > 1){ aCurrentWidth++; }
						aCurrentLine << aChar;
						if(aCurrentWidth > aLineWidth){
							aResString << aCurrentLine << "\n";
							aCurrentLine = "";
							aCurrentWidth = 0;
						}
					}
					aResString << aCurrentLine << "\n";
				}
				aResString.Delete(aResString.Length() - 1);
				
				this->Select(aLineFrom, aLineFrom);
				this->Delete(aLineFrom, aLineTo);
				this->Insert(aLineFrom, aResString.Text(), aResString.Length());
				this->Select(aLineFrom, aLineFrom + aResString.Length());
				delete[]	aText;
			}
			break;
			
		case K_TEXT_REMOVE_RETURNS:
			{
				char*	aText = new char[aLineTo - aLineFrom + 1];
				this->GetText(aLineFrom, aLineTo - aLineFrom, aText);
				
				for(int32 i = 0, j = 0; i <= aLineTo - aLineFrom; i++){
					if(*(aText + i) != LF){
						*(aText + j) = *(aText + i);
						j++;
					}
				}
				
				this->Select(aLineFrom, aLineFrom);
				this->Delete(aLineFrom, aLineTo);
				this->Insert(aLineFrom, aText, ::strlen(aText));
				this->Select(aLineFrom, aLineFrom + ::strlen(aText));
				delete[]	aText;
			}
			break;
			
		case K_HTML_TAG:
			const char	*pre, *post;
			msg->FindString("pretag", &pre);
			msg->FindString("posttag", &post);
			this->InsertTags(pre, post);
			break;
		
		case K_EMAIL_ADDRESS:
			{
				const char	*aEmail, *aName;
				char		*aText;
				
				msg->FindString("email_address", &aEmail);
				
				if(aFrom != aTo){//if some text is selected...
					aText = new char[::strlen(aEmail) + 256];
					::sprintf(aText, "<A HREF=\"mailto:%s\">", aEmail);
					this->InsertTags(aText, "</A>");
					delete[]	aText;
				}else if((aInfo.modifiers & B_COMMAND_KEY) != 0){//when the command key pressed...
					msg->FindString("name", &aName);
					aText = new char[::strlen(aEmail) + ::strlen(aName) + 256];
					::sprintf(aText, "<A HREF=\"mailto:%s\">%s</A>", aEmail, aName);
					this->Delete();
					this->Insert(aText);
					delete[]	aText;
				}else if((aInfo.modifiers & B_SHIFT_KEY) != 0){//when the shift key pressed...
					aText = new char[::strlen(aEmail) + 256];
					::sprintf(aText, "<A HREF=\"mailto:%s\">%s</A>", aEmail, aEmail);
					this->Delete();
					this->Insert(aText);
					delete[]	aText;
				}else{
					this->Delete();
					this->Insert(aEmail);
				}
			}
			break;
			
		case K_URL_STRING:
			{
				char		*aText;
				const char	*aURL, *aTitle;
				int32	aFrom, aTo;
				this->GetSelection(&aFrom, &aTo);
				
				msg->FindString("url_string", &aURL);
				
				if(aFrom != aTo){// if some text is selected...
					aText = new char[::strlen(aURL) + 256];
					::sprintf(aText, "<A HREF=\"%s\">", aURL);
					this->InsertTags(aText, "</A>");
					delete[]	aText;
					
				}else if((aInfo.modifiers & B_COMMAND_KEY) != 0){//when the command key pressed...
					if(msg->FindString("page_title", &aTitle) == B_NO_ERROR){
						aText = new char[::strlen(aURL) + ::strlen(aTitle) + 256];
						::sprintf(aText, "<A HREF=\"%s\">%s</A>", aURL, aTitle);
					}else{
						aText = new char[2 * ::strlen(aURL) + 256];
						::sprintf(aText, "<A HREF=\"%s\">%s</A>", aURL, aURL);
					}
					this->Delete();
					this->Insert(aText);
					delete[]	aText;
				}else if((aInfo.modifiers & B_SHIFT_KEY) != 0){//when the shift key pressed...
					aText = new char[2 * ::strlen(aURL) + 256];
					::sprintf(aText, "<A HREF=\"%s\">%s</A>", aURL, aURL);
					this->Delete();
					this->Insert(aText);
					delete[]	aText;
				}else{
					this->Delete();
					this->Insert(aURL);
				}
			}
			break;
			
		case K_TEXT_MODIFIED:
			if(this->DoesHTMLSyntax()){
				int32	from, to;
				msg->FindInt32("from_offset", &from);
				msg->FindInt32("to_offset", &to);
				this->UpdateHTMLSyntax(from, to);
				//::printf("modified\n");
			}
			break;
		
		default:
			if(msg->WasDropped()){
				this->WhenDropped(msg);
			}else{
				this->BTextView::MessageReceived(msg);
			}
	}
}


void
KTextView::KeyDown(const char* bytes, int32 numBytes)
{	
	BMessage*	msg = this->Window()->CurrentMessage();
	int32		aRawChar;
	uint32		aModifiers;
	msg->FindInt32("raw_char", &aRawChar);
	msg->FindInt32("modifiers", (int32*)&aModifiers);
	
	//::printf("raw_char = %ld, modifiers = %ld\n", aRawChar, aModifiers);
	
	if(!fShortcut->DoShortcut(aRawChar, aModifiers)){
		this->BTextView::KeyDown(bytes, numBytes);
	}
	if(fBackgroundBitmap != NULL){
		this->ClearViewBitmap();
		delete	fBackgroundBitmap;
		fBackgroundBitmap = NULL;
		this->Hide();
		this->Show();
	}
}


void
KTextView::FrameResized(float width, float height)
{
	if(this->DoesWordWrap()){
		BRect	rect(this->TextRect());
		rect.right = width;
		this->SetTextRect(rect);
	}
	
	this->BTextView::FrameResized(width, height);
}


void
KTextView::SetBaseFont(BFont* font)
{
	delete fBaseFont;
	fBaseFont = font;
	
	this->SetFontAndColor(0, this->TextLength(), fBaseFont, B_FONT_ALL, &fCharColor);
	if(this->DoesHTMLSyntax()){ this->UpdateHTMLSyntax(0, this->TextLength()); }
}


void
KTextView::Select(int32 from, int32 to)
{
	uint32	buttons = 0;
	BMessage*	msg = this->Window()->CurrentMessage();
	
	if(msg != NULL){
		msg->FindInt32("buttons", (int32*)&buttons);
	
		if(fMultiButton == K_COPY_PASTE && ((buttons & B_SECONDARY_MOUSE_BUTTON)!= 0)){
			BMessage* aMessage = new BMessage(B_COPY);
			int64	time = system_time();
			aMessage->AddInt64("when", time);
			this->Window()->PostMessage(aMessage);
		}
	}
	
	this->BTextView::Select(from, to);
}


void
KTextView::MouseDown(BPoint where)
{
	uint32	buttons;
	this->Window()->CurrentMessage()->FindInt32("buttons", (int32*)&buttons);
	key_info	info;
	get_key_info(&info);
	
	if(fMultiButton == K_SHOW_POPUP 
			&& (((buttons & B_SECONDARY_MOUSE_BUTTON)!= 0)
					|| (((buttons & B_PRIMARY_MOUSE_BUTTON)!= 0) && (info.modifiers & B_CONTROL_KEY) != 0))){
		BPopUpMenu*	popup = this->MakePopupMenu();
		BPoint	point(where);
		this->ConvertToScreen(&point);
		
		BMenuItem* menuItem = NULL;
		menuItem = popup->Go(point);
		
		if(menuItem != NULL){
			BMessage*	aMessage = menuItem->Message();
			if(aMessage != NULL)
				this->Window()->PostMessage(aMessage);
		}
		delete popup;
		
	}else{
		this->BTextView::MouseDown(where);
	}
	
	if(fMultiButton == K_COPY_PASTE && ((buttons & B_TERTIARY_MOUSE_BUTTON)!= 0)){
		this->Window()->PostMessage(new BMessage(B_PASTE));
	}
}


BPopUpMenu*
KTextView::MakePopupMenu()
{
	BPopUpMenu*	popup = new BPopUpMenu("popup");
	KMenuBar*	menubar = (KMenuBar*)this->Window()->FindView("menubar");
	KMenuUtils	utils;
	
	utils.AddMenuItem(popup, "Cut", B_CUT, this);
	utils.AddMenuItem(popup, "Copy", B_COPY, this);
	utils.AddMenuItem(popup, "Paste", B_PASTE, this);
	utils.AddMenuItem(popup, "Select All", B_SELECT_ALL, this);
	
	popup->AddSeparatorItem();
	
	//Tag menu
	menubar->ConstructTagMenu(popup);
	
	popup->AddSeparatorItem();
	//e-mail address menu
	utils.AddEmailAddressMenu(popup, K_EMAIL_ADDRESS, NULL, this->Window());
	//url menu
	utils.AddURLMenu(popup, K_URL_STRING, NULL, this->Window());
	
	//add-ons menu
	BMenu*	aMenu = new BMenu("Add-ons");
	menubar->MakeAddOnMenu(aMenu);
	if(aMenu->CountItems() == 0){
		delete aMenu;
	}else{
		popup->AddSeparatorItem();
		popup->AddItem(aMenu);
	}
	
	popup->AddSeparatorItem();
	utils.AddMenuItem(popup, "Undo", B_UNDO, this);
	
	return popup;
}


void
KTextView::InsertTags(const char* pre, const char* post)
{
	int32	from, to;
	this->GetSelection(&from, &to);
	
	int32	total = to - from + strlen(pre) + strlen(post);
	
	char*	text = new char[total + 1];
	strcpy(text, pre);
	this->GetText(from, to - from, text + strlen(pre));
	strcat(text, post);
	
	this->Delete(from, to);
	this->Insert(from, text, total);
	this->Select(from + total, from + total);
	
	delete []text;
}


void
KTextView::WhenDropped(BMessage* msg)
{	
	int32		num;
	uint32		type;
	char*		name;
	BPoint		aPoint;//raw drop point
	BPoint		point;//converted drop point
	entry_ref	ref;
	
	key_info	keyinfo;
	get_key_info(&keyinfo);
	
	msg->FindPoint("_drop_point_", &aPoint);
	point = aPoint;
	this->ConvertFromScreen(&point);

	const int32 offset = this->OffsetAt(point);
	
	/* now constructing... */
	if(msg->GetInfo(B_REF_TYPE, 0, &name, &type, &num) == B_NO_ERROR
			&& num >= 1){
		/* FILE */
		for(int32 i = 0; i < num; i++){
			entry_ref	ref;
			if(msg->FindRef("refs", i, &ref) != B_NO_ERROR) { continue; }
			this->FileDropped(msg, ref, offset, keyinfo);
		}
	}else if(msg->GetInfo(B_RGB_COLOR_TYPE, 0, &name, &type, &num) == B_NO_ERROR
			&& num >= 1){
		/* RGB COLOR message */
		rgb_color*	color;
		int32		size;
		if(msg->FindData(name, B_RGB_COLOR_TYPE, (const void**)&color, &size) != B_NO_ERROR){ return; }
		this->ColorDropped(msg, *color, offset, keyinfo);
	}else if(msg->FindString("be:url", 0, (const char**)&name) == B_NO_ERROR){
		/* URL message */
		this->URLDropped(msg, name, offset, keyinfo);
	}else{
		/* DEFAULT */
		this->BTextView::MessageReceived(msg);
		return;
	}
}


void
KTextView::MouseMoved(BPoint where, uint32 code, const BMessage* msg)
{
	key_info	keyinfo;
	get_key_info(&keyinfo);
	
	uint32	buttons;
	this->Window()->CurrentMessage()->FindInt32("buttons", (int32*)&buttons);
	
	if(msg != NULL){
		entry_ref	ref;
		const char*		buf;
		if(msg->FindRef("refs", &ref) == B_NO_ERROR){
			BFile	file(&ref, B_READ_ONLY);
			if(file.InitCheck() == B_NO_ERROR){
				BNodeInfo	info(&file);
				char		aFileType[B_MIME_TYPE_LENGTH];
				info.GetType(aFileType);
				if((::strncmp(aFileType, "text", 4) == 0)
						|| (::strncmp(aFileType, "image", 5) == 0)){
					int32 offset = this->OffsetAt(where);
					this->Select(offset, offset);
				}
			}
		}else if(msg->FindString("text/plain", &buf) == B_NO_ERROR
				|| (keyinfo.modifiers & B_OPTION_KEY) != 0
				|| ((buttons & B_TERTIARY_MOUSE_BUTTON) != 0
					&& this->GetMultiButtonFunction() == K_SHOW_POPUP)){
			int32	offset = OffsetAt(where);
			this->Select(offset, offset);
		}
	}
	
	this->BTextView::MouseMoved(where, code, msg);
}


void
KTextView::SetHTMLSyntax(bool status)
{
	KTextRunUtils	utils;
	if(status){
		this->UpdateHTMLSyntax(0, this->TextLength());
	}else{
		text_run_array array = utils.MakeTextRunArray(0, *fBaseFont, fCharColor);
		this->SetRunArray(0, this->TextLength(), &array);
	}
	fUseHTMLSyntax = status;
}


void
KTextView::UpdateHTMLSyntax(int32 from, int32 to)
{
	KTextRunUtils	utils;
	//rgb_color		curColor;
	//BFont			font;
	
	const char*	text = this->Text();
	const int32	len = this->TextLength();
	
	if(from != 0){
		for(int32 i = from - 1; i >= 0; i--){
			if(text[i] == '<'){
				from = i;
				break;
			}else if(text[i] == '>'){
				from = i + 1;
				break;
			}
		}
		//Sample
		for(int32 i = from - 1; i >= 0; i--){
			if(strncmp(text + i, "<!--", 4) == 0){
				from = i - 1;
				break;
			}else if(i >= 2 && strncmp(text + i - 2, "-->", 3) == 0){
				break;
			}
		}
	}
	for(int32 i = to; i <= len; i++){
		if(text[i] == '>' || text[i] == '<'){
			to = i;
			break;
		}
	}
	for(int32 i = to - 3; i <= len; i++){
		if(strncmp(text + i, "-->", 3) == 0){
			to = i + 3;
			break;
		}else if(strncmp(text + i, "<!--", 4) == 0){
			break;
		}
	}

	int32	aSelectionFrom, aSelectionTo;
	this->GetSelection(&aSelectionFrom, &aSelectionTo);
	this->Select(aSelectionFrom, aSelectionFrom);
	
	text_run_array	anArray = utils.MakeTextRunArray(0, fBaseFont, fCharColor);
	
	int32	fromOffset = from;
	for(int32 i = from; i <= to; i++){
		if(text[i] == '<' || i == len){
			text_run_array*	curArray = RunArray(fromOffset, i);
			if(curArray->count != 1
					|| !utils.IsSameColor(curArray->runs[0].color, fCharColor)){
				anArray.runs[0].color = fCharColor;
				this->SetRunArray(fromOffset, i, &anArray);
			}
			//for comment
			if(strncmp(text + i, "<!--", 4) == 0){
				int32	comstart = i;
				for(int32 j = i + 4; j <= to; j++){
					if(strncmp(text + j, "-->", 3) == 0){
						j += 3;
						curArray = RunArray(comstart, j);
						if(curArray->count != 1
								|| !utils.IsSameColor(curArray->runs[0].color, fCommentColor)){
							anArray.runs[0].color = fCommentColor;
							this->SetRunArray(comstart, j, &anArray);
						}
						i = j;
						break;
					}
				}
			}
			//free!
			free(curArray);
			fromOffset = i;
		}else if (text[i] == '>'){
			text_run_array*	curArray = RunArray(fromOffset, i + 1);
			if(i >= 2 && strncmp(text + i - 2, "-->", 3) != 0){
				if(curArray->count != 1
						|| !utils.IsSameColor(curArray->runs[0].color, fTagColor)){
					anArray.runs[0].color = fTagColor;
					SetRunArray(fromOffset, i + 1, &anArray);
				}
				fromOffset = i + 1;
			}
			//free
			free(curArray);
		}
	}
	
	this->Select(aSelectionFrom, aSelectionTo);
}


void
KTextView::FileDropped(BMessage* msg, entry_ref entry, int32 offset, key_info info)
{
	KTextUtils	util;
	char	aFileType[B_MIME_TYPE_LENGTH];
	BEntry	aEntry(&entry);
	BFile	aFile(&entry, B_READ_ONLY);
	KWindow*	aWindow = dynamic_cast<KWindow*>(this->Window());
	
	if(aWindow == NULL){ ::beep(); return; }
	if(aFile.InitCheck() != B_NO_ERROR){ ::beep(); return; }
	
	BNodeInfo	aInfo(&aFile);
	aInfo.GetType(aFileType);
	if(::strncmp(aFileType, "image", 5) == 0){
		/* IMAGE */
		BPath		aPath(&entry);
		BBitmap*	aBitmap = BTranslationUtils::GetBitmap(aPath.Path(), NULL);
		BEntry*		aDocEntry = aWindow->GetEntry();
		if(aBitmap != NULL && aDocEntry != NULL){
			BPath	aDocPath(aDocEntry);
			char*	aRelPath = util.GetRelPath(aDocPath.Path(), aPath.Path());
			BRect	aRect = aBitmap->Bounds();
			char*	aOutText = new char[256 + B_FILE_NAME_LENGTH];
			::sprintf(aOutText, "<IMG SRC=\"%s\" WIDTH=\"%ld\" HEIGHT=\"%ld\" ALT=\"\">"
						, aRelPath, (int32)(aRect.Width() + 1), (int32)(aRect.Height() + 1));
			this->Insert(offset, aOutText, ::strlen(aOutText));
			this->Select(offset, offset + ::strlen(aOutText));
			delete[]	aOutText;
			delete[]	aRelPath;
		}else{
			::beep();
			(new BAlert("caution :"
						, "The image file is invalid or this document hasn't been saved.\n"
						"Check them and try again."
						, "OK"
						, NULL, NULL
						, B_WIDTH_AS_USUAL, B_WARNING_ALERT))->Go();
		}
		delete	aBitmap;
	}else if(::strncmp(aFileType, "text", 4) == 0){
		/* TEXT */
		if((info.modifiers & B_COMMAND_KEY) != 0
				|| (info.modifiers & B_OPTION_KEY) != 0){
			BEntry*		aDocEntry = aWindow->GetEntry();
			if(aDocEntry != NULL){
				BPath	aPath(&entry);
				BPath	aDocPath(aDocEntry);
				char*	aRelPath = util.GetRelPath(aDocPath.Path(), aPath.Path());
				this->Insert(offset, aRelPath, ::strlen(aRelPath));
				this->Select(offset, offset + ::strlen(aRelPath));
				delete[]	aRelPath;
			}
		}else{
			off_t	aSize;
			aFile.GetSize(&aSize);
			char*	aText = new char[aSize + 1];
			aText[aSize] = '\0';
			aFile.Lock();
			aFile.Read(aText, aSize);
			util.ConvertReturnsToLF(aText);
			util.ConvertCharCodeToUTF8(&aText);
			aSize = ::strlen(aText);
			this->Insert(offset, aText, aSize);
			this->Select(offset, offset + aSize);
			aFile.Unlock();
			delete[]	aText;
		}
	}else if(::strcmp(aFileType, "application/x-vnd.Be-bookmark") == 0){
		/* Bookmark */
		attr_info	aTitleAttrInfo;
		attr_info	aURLAttrInfo;
		if((aFile.GetAttrInfo("META:title", &aTitleAttrInfo) != B_NO_ERROR)
				|| (aFile.GetAttrInfo("META:url", &aURLAttrInfo) != B_NO_ERROR)
				|| (aTitleAttrInfo.size <= 1) || (aURLAttrInfo.size <= 1)){ ::beep(); return; }
		char*	aTitle = new char[aTitleAttrInfo.size];
		char*	aURL = new char[aURLAttrInfo.size];
		aFile.ReadAttr("META:title", B_STRING_TYPE, 0, aTitle, aTitleAttrInfo.size);
		aFile.ReadAttr("META:url", B_STRING_TYPE, 0, aURL, aURLAttrInfo.size);
		
		BMessage	aMessage(K_URL_STRING);
		aMessage.AddString("page_title", aTitle);
		aMessage.AddString("url_string", aURL);
		this->MessageReceived(&aMessage);
		
		delete[]	aURL;
		delete[]	aTitle;
	}else if(::strcmp(aFileType, "application/x-person") == 0){
		/* Person */
		// do nothing...
	}
}

void
KTextView::ColorDropped(BMessage* msg, rgb_color color, int32 offset, key_info info)
{
	if((info.modifiers & B_COMMAND_KEY) != 0){ // ALT key pressed...
		this->ModifyColorPrefs(msg);
	}else{
		this->BTextView::MessageReceived(msg);
	}
}

void
KTextView::URLDropped(BMessage* msg, const char* name, int32 offset, key_info info)
{
	BMessage aMessage(K_URL_STRING);
	aMessage.AddString("url_string", name);
	this->MessageReceived(&aMessage);
}



void
KTextView::ModifyColorPrefs(BMessage* msg)
{
	char*	aColorName;
	uint32	aType;
	int32	aNum;
	BPoint	aPoint;
	
	/* Get the drop point */
	msg->FindPoint("_drop_point_", &aPoint);
		
	/* Get RGB Color Information */
	msg->GetInfo(B_RGB_COLOR_TYPE, 0, &aColorName, &aType, &aNum);
	
	KMenuUtils	aUtils;
	rgb_color*	aColor;
	int32		aSize;
	/* Get RGB Color */
	msg->FindData(aColorName, B_RGB_COLOR_TYPE, (const void **) &aColor, &aSize);
	/* COLOR MENU */
	BPopUpMenu*	aPopUp = new BPopUpMenu("");
	//aUtils.AddMenuItem(aPopUp, "Insert RGB Color", K_INSERT_COLOR, this);
	//aPopUp->AddSeparatorItem();
	aUtils.AddMenuItem(aPopUp, "Set Background Color", K_SET_BG_COLOR, this);
	aUtils.AddMenuItem(aPopUp, "Set Character Color", K_SET_CHR_COLOR, this);
	aUtils.AddMenuItem(aPopUp, "Set Tag Color", K_SET_TAG_COLOR, this);
	aUtils.AddMenuItem(aPopUp, "Set Comment Color", K_SET_COM_COLOR, this);
	BMenuItem*	aMenuItem = NULL;
	/* Show PopUpMenu */
	aMenuItem = aPopUp->Go(aPoint, false, true, this->Frame());
	
	if(aMenuItem != NULL){
		BMessage*	aMessage = aMenuItem->Message();
		KPrefs*		aPrefs = ((KApp*)be_app)->GetPrefs();
		
		/* Set Colors */
		switch(aMessage->what){
			case K_SET_BG_COLOR:
				fBackgroundColor = *aColor;
				this->SetViewColor(fBackgroundColor);
				this->Invalidate(this->Bounds());
				aPrefs->SetData("background_color", &fBackgroundColor);
				break;
			case K_SET_CHR_COLOR:
				fCharColor = *aColor;
				this->UpdateHTMLSyntax(0, this->TextLength());
				aPrefs->SetData("char_color", &fCharColor);
				break;
			case K_SET_TAG_COLOR:
				fTagColor = *aColor;
				this->UpdateHTMLSyntax(0, this->TextLength());
				aPrefs->SetData("tag_color", &fTagColor);
				break;
			case K_SET_COM_COLOR:
				fCommentColor = *aColor;
				this->UpdateHTMLSyntax(0, this->TextLength());
				aPrefs->SetData("comment_color", &fCommentColor);
				break;
			default:
				;
		}
	}
}





























