#ifndef __KString_H__
#define __KString_H__

#include <List.h>
#include <SupportDefs.h>


/*----------constants---------*/
const uint8		kLF = 10;

enum{
	K_NO_STRING_LIMIT	= -1
};


struct	text_item{
	int32	from;
	int32	length;
};

class	BList;


class KString
{
public:
	KString(const char* text
				, int32 len = K_NO_STRING_LIMIT
				, bool will_reallocate_memory = true);	/* KString s = "aaa"; */
	KString();											/* KString s; */
	KString(const KString& string
				, int32 len = K_NO_STRING_LIMIT
				, bool will_reallocate_memory = true);	/* KString s = KString...; */
	~KString();
	
	/* check */
	bool		IsEmpty() const;
	bool		IsEditable() const;
	
	/* data reset */
	KString&	SetTo(const char* text
						, int32 len = K_NO_STRING_LIMIT
						, bool will_reallocate_memory = true);
	KString&	SetTo(const KString& string
						, int32 len = K_NO_STRING_LIMIT
						, bool will_reallocate_memory = true);
	
	/* data handling */
	KString&	Append(const char* text, int32 len = K_NO_STRING_LIMIT);
	KString&	Append(const KString& string, int32 len = K_NO_STRING_LIMIT);
	KString&	Prepend(const char* text, int32 len = K_NO_STRING_LIMIT);
	KString&	Prepend(const KString& string, int32 len = K_NO_STRING_LIMIT);
	KString&	Insert(const char* text, int32 position, int32 len = K_NO_STRING_LIMIT);
	KString&	Insert(const KString& string, int32 position, int32 len = K_NO_STRING_LIMIT);
	KString&	Delete();
	KString&	Delete(int32 from, int32 to = K_NO_STRING_LIMIT);
	KString&	Copy(const char* text, int32 offset, int32 len = K_NO_STRING_LIMIT);
	KString&	Copy(const KString& string, int32 offset, int32 len = K_NO_STRING_LIMIT);
	
	
	/* data access */
	char*		Text() const;
	int32		Length() const;
	
	/* character */
	KString		Character(int32);
	KString		Characters(int32 from, int32 to);
	int32		CountCharacters() const;
	
	/* word */
	KString		Word(int32);
	KString		Words(int32 from, int32 to);
	int32		CountWords();
	
	BList		WordList();
	
	int32		OffsetOfWord(int32);
	
	/* paragraph */
	KString		Paragraph(int32 index);
	KString		Paragraphs(int32 from, int32 to);
	int32		CountParagraphs();
	
	BList		ParagraphList();
	
	/* text item */
	KString		TextItem(int32);
	KString		TextItems(int32 from, int32 to);
	int32		CountTextItems();
	
	BList		TextItemList();
	
	void		SetTextItemDelimiter(const char*); /* default : tab */
	char*		TextItemDelimiter() const;
	
	/* compare */
	int32			Compare(const char* text, bool ignore_case = true
							, int32 len = K_NO_STRING_LIMIT) const;
	int32			Compare(const KString& string, bool ignore_case = true
							, int32 len = K_NO_STRING_LIMIT) const;
	static int32	Compare(const char* textA, const char* textB, bool ignore_case = true
							, int32 len = K_NO_STRING_LIMIT);
	static int32	Compare(const KString& stringA, const KString& stringB, bool ignore_case = true
							, int32 len = K_NO_STRING_LIMIT);
	
	bool		Is(const char* text, bool ignore_case = true) const;
	bool		Is(const KString& text, bool ignore_case = true) const;
	
	bool		StartsWith(const char* text, bool ignore_case = true) const;
	bool		StartsWith(const KString& string, bool ignore_case = true) const;
	
	bool		EndsWith(const char* text, bool ignore_case = true) const;
	bool		EndsWith(const KString& string, bool ignore_case = true) const;
	
	bool		Contains(const char* text
						, bool ignore_case = true
						, int32* from = NULL
						, int32* to = NULL) const;
	bool		Contains(const KString& string
						, bool ignore_case = true
						, int32* from = NULL
						, int32* to = NULL) const;
	
	
	/* replace */					
	KString&	ReplaceAll(const char* search_text
							, const char* replace_text
							, bool ignore_case = true
							, int32* count = NULL);
	
	/* case */
	KString&	ToLower();
	KString&	ToUpper();
	
	/* direct edit */
	char*		BeginDirectEdit();
	KString&	EndDirectEdit();
	
	/* debug */
	void		PrintToStream();
	
	/* input operators */
	KString&	operator<<(const char*);
	KString&	operator<<(const KString&);
	KString&	operator<<(uint8);
	KString&	operator<<(uint32);
	KString&	operator<<(uint64);
	KString&	operator<<(int32);
	KString&	operator<<(int64);
	KString&	operator<<(float);/* = %.2f */
	
	/* operators */
	KString&	operator=(const char*);
	KString&	operator=(const KString&);
	KString&	operator+(const char*);
	KString&	operator+(const KString&);
	KString&	operator+=(const char*);
	KString&	operator+=(const KString&);
	char&		operator[](int32);
	const char&	operator[](int32) const;
	bool		operator==(const KString&);
	bool		operator==(const char*);
	bool		operator!=(const KString&);
	bool		operator!=(const char*);
	friend bool	operator==(const KString&, const char*);
	friend bool	operator==(const KString&, const KString&);
	friend bool	operator!=(const KString&, const char*);
	friend bool	operator!=(const KString&, const KString&);
	
private:
	
	char*	fString;
	char*	fDelimiter;
	
	BList	fWordList;
	BList	fParagraphList;
	BList	fTextItemList;
	
	
	int32	fCurrentVer;
	int32	fWordVer;
	int32	fParagVer;
	int32	fTextItemVer;
	bool	fReallocMem;
	
	bool	IsASCII(const char* aChar) const;
	bool	IsTopOfChar(const char* aChar) const;
	bool	IsWordChar(const char* aChar) const;
	
	void	AnalyseWord();
	void	AnalyseParagraph();
	void	AnalyseTextItem();
	void	Modify();
};


/*------Inline definitions----------*/


inline char*	KString::Text() const { return fString; }
inline int32	KString::Length() const { return ::strlen(fString); }
inline bool		KString::IsTopOfChar(const char* aChar) const { return ((*aChar & 0xc0) != 0x80); }
inline bool		KString::IsASCII(const char* aChar) const { return ((*aChar & 0x80) == 0); }
inline void		KString::Modify(){ fCurrentVer++; }
inline bool		KString::IsEditable() const { return fReallocMem; }
inline char*	KString::BeginDirectEdit(){ return fString; }
inline KString&	KString::EndDirectEdit(){ this->Modify(); return *this; }
inline char*	KString::TextItemDelimiter() const { return fDelimiter; }


inline bool 	KString::operator==(const KString& string){ return (::strcmp(fString, string.Text()) == 0); }
inline bool 	KString::operator==(const char* text){ return (::strcmp(fString, text) == 0); }
inline bool 	KString::operator!=(const KString& string){ return (::strcmp(fString, string.Text()) != 0); }
inline bool 	KString::operator!=(const char* text){ return (::strcmp(fString, text) != 0); }
inline bool 	operator==(const KString& string, const char* text){ return (::strcmp(string.Text(), text) == 0); }
inline bool 	operator==(const KString& stringA, const KString& stringB){ return (::strcmp(stringA.Text(), stringB.Text()) == 0); }
inline bool 	operator!=(const KString& string, const char* text){ return (::strcmp(string.Text(), text) != 0); }
inline bool 	operator!=(const KString& stringA, const KString& stringB) { return (::strcmp(stringA.Text(), stringB.Text()) != 0); }

inline char& 		KString::operator[](int32 index){ return fString[index]; }
inline const char&	KString::operator[](int32 index) const { return fString[index]; }

inline int32
KString::Compare(const char* textA, const char* textB, bool ignore_case, int32 len)
{
	int32	aLen = len;
	char	c1, c2;
	int32	aLenA = ::strlen(textA);
	int32	aLenB = ::strlen(textB);
	if(len == K_NO_STRING_LIMIT){ aLen = (aLenA > aLenB ? aLenA : aLenB); }
	for(int32 i = 0; i < aLen; i++){
		c1 = *(textA + i);
		c2 = *(textB + i);
		if(ignore_case){
			if(c1 >= 'A' && c1 <= 'Z'){ c1 += 'a' - 'A'; }
			if(c2 >= 'A' && c2 <= 'Z'){ c2 += 'a' - 'A'; }
		}
		if( c1 < c2 ){
			return -1;
		}else if( c1 > c2 || c2 == '\0'){
			return 1;
		}
	}
	return 0;
}

inline int32
KString::CountCharacters() const
{
	int32	aNum = 0;
	for(int32 i = 0;; i++){
		if(*(fString + i) == '\0') { break; }
		if(this->IsTopOfChar(fString + i)) { aNum++; }
	}
	return aNum;
}

inline bool
KString::IsEmpty() const
{
	if(*fString != '\0'){ return false; }
	return true;
}


inline bool
KString::IsWordChar(const char* aChar) const
{
	if((*aChar >= 'a' && *aChar <= 'z')
			|| (*aChar >= 'A' && *aChar <= 'Z')
			|| *aChar == '_'){ return true; }

	return false;
}



#endif











