/* $Id: GLView.cpp,v 1.2 2000/03/19 01:13:13 brianp Exp $ */

/*
 * Mesa 3-D graphics library
 * Version:  3.1
 * 
 * Copyright (C) 1999  Brian Paul   All Rights Reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * BRIAN PAUL BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
 * AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */


/*
 * $Log: GLView.cpp,v $
 * Revision 1.2  2000/03/19 01:13:13  brianp
 * updated for Mesa 3.3
 *
 * Revision 1.1.1.1  1999/08/19 00:55:41  jtg
 * Imported sources
 *
 * Revision 1.7  1999/03/28 21:08:17  brianp
 * updated SetBuffer driver function
 *
 * Revision 1.6  1999/02/14 03:44:37  brianp
 * new copyright
 *
 * Revision 1.5  1999/02/11 03:50:57  brianp
 * added CopySubBufferMESA()
 *
 * Revision 1.4  1999/02/06 17:44:59  brianp
 * code clean-up and bug fixes
 *
 * Revision 1.3  1999/02/04 04:13:15  brianp
 * implemented double buffering
 *
 * Revision 1.2  1999/02/03 04:23:28  brianp
 * basic device driver functions now work (yeah!)
 *
 * Revision 1.1  1999/02/02 04:40:46  brianp
 * Initial revision
 */



#include <assert.h>
#include <stdio.h>
#include <GLView.h>
#include "../src/context.h"
#include "../src/buffers.h"

/* nVidia 3D add-on */
#include "nv_globals.h"
#include "nv_3da.h"
#include "riva_glh.h"
#include "glh.h"
#include "riva_ctx.h"
#include "riva_symbols.h"

// BeOS component ordering for B_RGBA32 bitmap format
#define BE_RCOMP 2
#define BE_GCOMP 1
#define BE_BCOMP 0
#define BE_ACOMP 3

//
// This object hangs off of the BGLView object.  We have to use
// Be's BGLView class as-is to maintain binary compatibility (we
// can't add new members to it).  Instead we just put all our data
// in this class and use BGLVIew::m_gc to point to it.
//
class AuxInfo
{
public:
   AuxInfo();
   ~AuxInfo();
   void Init(BGLView *bglView, GLcontext *c, GLvisual *v, GLframebuffer *b);

   void MakeCurrent();
   void SwapBuffers(bool finish) const;
   void CopySubBuffer(GLint x, GLint y, GLuint width, GLuint height) const;
   void Resize(GLuint width, GLuint height);

private:
   AuxInfo(const AuxInfo &rhs);  // copy constructor illegal
   AuxInfo &operator=(const AuxInfo &rhs);  // assignment oper. illegal

   GLcontext *mContext;
   GLvisual *mVisual;
   GLframebuffer *mBuffer;

   BGLView *mBGLView;

   GLubyte mColor[4];       // current color
   GLuint mIndex;           // current color index
   GLubyte mClearColor[4];  // buffer clear color
   GLuint mClearIndex;      // buffer clear color index
   GLint mBottom;           // used for flipping Y coords
   GLint mUnalignedWidth;	// 'real' colorbuffer width (excl. engine slopspace)
   GLint mAlignedWidth;		// colorbuffer width including engine slopspace
   GLint mHeight;   		// colorbuffer height
   bool mBitmap;			// true if colorbuffer initialized 

   // Mesa device driver functions
   static void UpdateState(GLcontext *ctx);
   static void ClearIndex(GLcontext *ctx, GLuint index);
   static void ClearColor(GLcontext *ctx, GLubyte r, GLubyte g,
                          GLubyte b, GLubyte a);
   static GLbitfield Clear(GLcontext *ctx, GLbitfield mask,
                                GLboolean all, GLint x, GLint y,
                                GLint width, GLint height);
   static void ClearFront(GLcontext *ctx, GLboolean all, GLint x, GLint y,
                          GLint width, GLint height);
   static void ClearBack(GLcontext *ctx, GLboolean all, GLint x, GLint y,
                         GLint width, GLint height);
   static void Index(GLcontext *ctx, GLuint index);
   static void Color(GLcontext *ctx, GLubyte r, GLubyte g,
                     GLubyte b, GLubyte a);
   static GLboolean SetDrawBuffer(GLcontext *ctx, GLenum mode);
   static void SetReadBuffer(GLcontext *ctx, GLframebuffer *colorBuffer,
                             GLenum mode);
   static void GetBufferSize(GLcontext *ctgx, GLuint *width,
                             GLuint *height);
   static const GLubyte *GetString(GLcontext *ctx, GLenum name);

   // Front-buffer functions
   static void WriteRGBASpanFront(const GLcontext *ctx, GLuint n,
                                  GLint x, GLint y,
                                  CONST GLubyte rgba[][4],
                                  const GLubyte mask[]);
   static void WriteRGBSpanFront(const GLcontext *ctx, GLuint n,
                                 GLint x, GLint y,
                                 CONST GLubyte rgba[][3],
                                 const GLubyte mask[]);
   static void WriteMonoRGBASpanFront(const GLcontext *ctx, GLuint n,
                                      GLint x, GLint y, const GLubyte mask[]);
   static void WriteRGBAPixelsFront(const GLcontext *ctx, GLuint n,
                                    const GLint x[], const GLint y[],
                                    CONST GLubyte rgba[][4],
                                    const GLubyte mask[]);
   static void WriteMonoRGBAPixelsFront(const GLcontext *ctx, GLuint n,
                                        const GLint x[], const GLint y[],
                                        const GLubyte mask[]);
   static void WriteCI32SpanFront(const GLcontext *ctx, GLuint n,
                                  GLint x, GLint y,
                                  const GLuint index[], const GLubyte mask[]);
   static void WriteCI8SpanFront(const GLcontext *ctx, GLuint n,
                                 GLint x, GLint y,
                                 const GLubyte index[], const GLubyte mask[]);
   static void WriteMonoCISpanFront(const GLcontext *ctx, GLuint n,
                                    GLint x, GLint y, const GLubyte mask[]);
   static void WriteCI32PixelsFront(const GLcontext *ctx,
                                    GLuint n, const GLint x[], const GLint y[],
                                    const GLuint index[], const GLubyte mask[]);
   static void WriteMonoCIPixelsFront(const GLcontext *ctx, GLuint n,
                                      const GLint x[], const GLint y[],
                                      const GLubyte mask[]);
   static void ReadCI32SpanFront(const GLcontext *ctx,
                                 GLuint n, GLint x, GLint y, GLuint index[]);
   static void ReadRGBASpanFront(const GLcontext *ctx, GLuint n,
                                 GLint x, GLint y,
                                 GLubyte rgba[][4]);
   static void ReadCI32PixelsFront(const GLcontext *ctx,
                                   GLuint n, const GLint x[], const GLint y[],
                                   GLuint indx[], const GLubyte mask[]);
   static void ReadRGBAPixelsFront(const GLcontext *ctx,
                                   GLuint n, const GLint x[], const GLint y[],
                                   GLubyte rgba[][4], const GLubyte mask[]);

   // Back buffer functions
   static void WriteRGBASpanBack(const GLcontext *ctx, GLuint n,
                                  GLint x, GLint y,
                                  CONST GLubyte rgba[][4],
                                  const GLubyte mask[]);
   static void WriteRGBSpanBack(const GLcontext *ctx, GLuint n,
                                 GLint x, GLint y,
                                 CONST GLubyte rgba[][3],
                                 const GLubyte mask[]);
   static void WriteMonoRGBASpanBack(const GLcontext *ctx, GLuint n,
                                      GLint x, GLint y, const GLubyte mask[]);
   static void WriteRGBAPixelsBack(const GLcontext *ctx, GLuint n,
                                    const GLint x[], const GLint y[],
                                    CONST GLubyte rgba[][4],
                                    const GLubyte mask[]);
   static void WriteMonoRGBAPixelsBack(const GLcontext *ctx, GLuint n,
                                        const GLint x[], const GLint y[],
                                        const GLubyte mask[]);
   static void WriteCI32SpanBack(const GLcontext *ctx, GLuint n,
                                 GLint x, GLint y,
                                 const GLuint index[], const GLubyte mask[]);
   static void WriteCI8SpanBack(const GLcontext *ctx, GLuint n, GLint x, GLint y,
                                const GLubyte index[], const GLubyte mask[]);
   static void WriteMonoCISpanBack(const GLcontext *ctx, GLuint n,
                                   GLint x, GLint y,
                                   const GLubyte mask[]);
   static void WriteCI32PixelsBack(const GLcontext *ctx,
                                   GLuint n, const GLint x[], const GLint y[],
                                   const GLuint index[], const GLubyte mask[]);
   static void WriteMonoCIPixelsBack(const GLcontext *ctx,
                                     GLuint n, const GLint x[], const GLint y[],
                                     const GLubyte mask[]);
   static void ReadCI32SpanBack(const GLcontext *ctx,
                                GLuint n, GLint x, GLint y, GLuint index[]);
   static void ReadRGBASpanBack(const GLcontext *ctx, GLuint n,
                                GLint x, GLint y,
                                GLubyte rgba[][4]);
   static void ReadCI32PixelsBack(const GLcontext *ctx,
                                  GLuint n, const GLint x[], const GLint y[],
                                  GLuint indx[], const GLubyte mask[]);
   static void ReadRGBAPixelsBack(const GLcontext *ctx,
                                  GLuint n, const GLint x[], const GLint y[],
                                  GLubyte rgba[][4], const GLubyte mask[]);

};



AuxInfo::AuxInfo()
{
   mContext = NULL;
   mVisual = NULL;
   mBuffer = NULL;
   mBGLView = NULL;
   mBitmap = false;
   mClearColor[BE_RCOMP] = 0;
   mClearColor[BE_GCOMP] = 0;
   mClearColor[BE_BCOMP] = 0;
   mClearColor[BE_ACOMP] = 0;
   mClearIndex = 0;
   mColor[BE_RCOMP] = 255;
   mColor[BE_GCOMP] = 255;
   mColor[BE_BCOMP] = 255;
   mColor[BE_ACOMP] = 255;
   mIndex = 1;
}


AuxInfo::~AuxInfo()
{

   gl_destroy_visual(mVisual);
   if (mBuffer->DepthBuffer)
   {
       mBuffer->DepthBuffer = NULL;
   }
   gl_destroy_framebuffer(mBuffer);
   gl_destroy_context(mContext);
}


void AuxInfo::Init(BGLView *bglView, GLcontext *c, GLvisual *v, GLframebuffer *b)
{
   mBGLView = bglView;
   mContext = c;
   mVisual = v;
   mBuffer = b;
}


void AuxInfo::MakeCurrent()
{
   UpdateState(mContext);
   gl_make_current(mContext, mBuffer);
}

/* note: we should probably implement triple buffering. an intermediate buffer would
 * be handy to update from when the output window is moved (partly) offscreen, and
 * then moved back onto the screen again. (Be's OGL lib uses triple buffering.) */
/* note also:
 * Could be that a triple buffering scheme, with only updating the FrontBuffer via
 * draw() fixes our cliplist and positioning trouble in non-direct mode... */
void AuxInfo::SwapBuffers(bool finish) const
{
	/* instruct MESA to flush all pending render commands */
	/* note: this routine may only be called right before a scheduled swap! */
	if (finish) _mesa_swapbuffers(mContext);

	/* we need to make sure our outputwindow is fully constructed and operational! */
	static bool first_swap = true;
	if (first_swap)
	{
		snooze(20000);
		first_swap = false;
	}

	/* we need to wait for the engine to become idle at least once per frame,
	 * otherwise the system will come to a standstill!
	 * Note please that the best moment to wait for the engine is here: right before
	 * the 2D blit, so NOT after it (saves upto 10% speed). */
	//fixme: note that the 2D driver should best also have the sync_to_token thing now!
	RivaSync(0);

	/* HACK alert #1!! BGLView's DirectConnected() clip_list is broken! (checked dano):
	 * using a menubar offset calculation workaround. Seems to work well.. */
	if (mBitmap)
	{
		if(DirectMode)
		{
			LOG(2,("AuxInfo::Swapbuffers: in DirectMode\n"));

			/* must be outside aquired drawing_lock sem and outside DirectConnected! */
			mBGLView->GetClippingRegion(&dvi);

			/* prevent us waiting forever on a semaphore if the workspace has changed:
			 * so make sure we are visible first */
			/* note: (hack alert)
			 * also check view's count because upon app shut-down
			 * BGLView::DirectConnected() with B_DIRECT_STOP is called too late
			 * to block rendering before the background updates! */
			if (fNumClipRects && dvi.CountRects())
			{
				/* get the right to do direct screen access. */
				/* Note:
				 * we have to get the semaphore here because we want dbi to be valid */
				if (acquire_sem(drawing_lock) == B_NO_ERROR)
				{
					/* determine if we need to offset our clipping calculations because of
					 * an existing BMenuBar: doing this once on startup.. */
					if (!menu_offset_done)
					{
						/* find view's left-top offset from the output window */
						if (mBGLView->Window()->KeyMenuBar())
						{
							/* assuming the vertical size is fixed:
							 * Can the actual value be retreived somehow? If so, use that.. */
							//rudolf: the 3Dlife app gives correct offset in Frontbuffer.
							//(see riva_ctx.cpp, RivaSwapBuffers())
							//so: hoping for the best in compatibility. :-/ (or: assuming fault in directmode only.)
							menu_offset = 20;
							LOG(4,("AuxInfo::Swapbuffers 'init': Menubar detected, top offset from windows is %d\n",
								menu_offset));
						}
						else
						{
							menu_offset = 0;
							LOG(4,("AuxInfo::Swapbuffers 'init': No menubar detected, top offset from window is 0\n"));
						}
						menu_offset_done = true;

						/* update the cliprects! */
						LOG(4,("Swapbuffers 'init' # cliprects before fix %d\n", fNumClipRects));
						for (int32 cnt = 0; cnt < fNumClipRects; cnt++)
						{
							LOG(4,("cliplist[%d] before fix %d,%d;%d,%d\n", cnt,
								fClipList[cnt].left, fClipList[cnt].top,
								fClipList[cnt].right, fClipList[cnt].bottom));
						}
						clipping_rect * src = fClipList;
						clipping_rect * dst = &fClipList[fNumClipRects];
						uint32 old_cnt = fNumClipRects;
						for (uint32 cnt = 0; cnt < old_cnt; cnt++)
						{
							if (src->top < (fWindowBounds.top + menu_offset))
								src->top = (fWindowBounds.top + menu_offset);
							if (src->top <= src->bottom)
							{
								src++;
							}
							else
							{
								*dst--;
								*src = *dst;
								fNumClipRects--;
							}
						}
						LOG(4,("Swapbuffers 'init': # cliprects after fix %d\n", fNumClipRects));
						for (int32 cnt = 0; cnt < fNumClipRects; cnt++)
						{
							LOG(4,("cliplist[%d] after fix %d,%d;%d,%d\n", cnt,
								fClipList[cnt].left, fClipList[cnt].top,
								fClipList[cnt].right, fClipList[cnt].bottom));
						}
					}
					/* sync to vertical retrace if requested by the user */
					if (si->settings.force_sync) RivaWaitRetrace();
					/* blit the backbuffer to the frontbuffer */
				 	RivaSwapBuffersDirect();
					/* release the direct screen access */
					release_sem(drawing_lock);
				}
			}
			/* reset driver if needed */
			if ((si->engine.threeD.newmode & clone_nr) && !driver_updated) nvUpdateBuffers(true);
		}
		else
		{
			/* HACK alert #2!! we don't have access to dbi now at all!! */
			/* note: using Invalidate() combined with Draw() does NOT fix the
			 * rubbish issues. We need to loose non-direct mode support! */
			LOG(2,("AuxInfo::Swapbuffers: not in DirectMode\n"));

			if (!fNumClipRects)
			{
				/* get the current left-top position of the output window */
				WindowFrame = mBGLView->Window()->Frame();
				/* must be outside aquired drawing_lock sem and outside DirectConnected! */
				mBGLView->GetClippingRegion(&dvi);

				if (!menu_offset_done)
				{
					/* find view's left-top offset from the output window */
					/* note:
					 * apparantly non-direct mode doesn't need this workaround, but
					 * direct-mode does (we might be switching to direct-mode later!)
					 * Checked 3Dlife as a non-direct mode app: it's working OK without this
					 * hack.
					 * So menu_offset is only actually used in RivaSwapBuffersDirect(). */
					if (mBGLView->Window()->KeyMenuBar())
					{
						/* assuming the vertical size is fixed:
						 * Can the actual value be retreived somehow? If so, use that.. */
						menu_offset = 20;
						LOG(4,("AuxInfo::Swapbuffers 'init': Menubar detected, top offset from windows is %d\n",
							menu_offset));
					}
					else
					{
						menu_offset = 0;
						LOG(4,("AuxInfo::Swapbuffers 'init': No menubar detected, top offset from window is 0\n"));
					}
					menu_offset_done = true;
				}
				/* sync to vertical retrace if requested by the user */
				if (si->settings.force_sync) RivaWaitRetrace();
				/* blit the backbuffer to the frontbuffer */
				RivaSwapBuffersIndirect();
			}
			/* reset driver if needed */
			if ((si->engine.threeD.newmode & clone_nr) && !si->engine.threeD.mode_changing)
				nvUpdateBuffers(true);
		}
	}
}

//rudolf: if we want to keep this even though we have HWacc, then fix.
//otherwise: loose.
void AuxInfo::CopySubBuffer(GLint x, GLint y, GLuint width, GLuint height) const
{
	LOG(8,("AuxInfo::CopySubBuffer: called, but not impl. OK yet!!!\n"));

   if (mBitmap)
   {
      RivaSync(0);

      // Source bitmap and view's bitmap are same size.
      // Source and dest rectangle are the same.
      // Note (x,y) = (0,0) is the lower-left corner, have to flip Y
      BRect srcAndDest;
      srcAndDest.left = x;
      srcAndDest.right = x + width - 1;
      srcAndDest.bottom = mBottom - y;
      srcAndDest.top = srcAndDest.bottom - height + 1;

  	  //rudolf:
      //need hardware function if we keep CopySubBuffer, BBitmaps are nolonger used: 
      //mBGLView->DrawBitmap(mBitmap, srcAndDest, srcAndDest);
   }
}

void AuxInfo::Resize(GLuint width, GLuint height)
{
	/* block driver rendering */
	AQUIRE_BEN(si->engine.lock)
	nv_start_dma();
	nv_acc_wait_idle_dma();
	si->engine.threeD.newmode |= clone_nr;
	RELEASE_BEN(si->engine.lock)

	/* set new buffer sizes */
	FrontBuffer.right = FrontBuffer.left + (width - 1);
	FrontBuffer.bottom = FrontBuffer.top + (height - 1);

	/* cleanup sometime:
	 * this variable could be used throughout the driver, instead of using
	 * (FrontBuffer.right - FrontBuffer.left + 1) */
	mUnalignedWidth = width;
	mBottom = (GLint) FrontBuffer.bottom;

	/* we have to account for engine granularity!!! (NV18: 64 bytes) */
	switch(si->dm.space)
	{
	case B_RGB15_LITTLE:
	case B_RGB16_LITTLE:
		width = (width + 0x1f) & ~0x1f;
		break;
	case B_RGB32_LITTLE:
		width = (width + 0x0f) & ~0x0f;
		break;
	default:
		width = (width + 0x3f) & ~0x3f;
		break;
	}
	mAlignedWidth = width;

	/* tell mesa to update it's buffers */
	//not needed, apps should call glViewport() afterwards!
	//_mesa_ResizeBuffersMESA();

	/* tell hardware to update it's buffers */
	riva_sinfo[0].SetSurfaces3D(
		((rivaContext.bufRender == GL_FRONT) ?
			riva_sinfo[0].frontbufferoffset :
			riva_sinfo[0].backbufferoffset),
		riva_sinfo[0].depthbufferoffset);

	/* make sure 3D state will be reloaded */
	si->engine.threeD.reload |= clone_nr;
	/* signal OK to start rendering again */
	si->engine.threeD.newmode &= ~clone_nr;
}

void AuxInfo::UpdateState( GLcontext *ctx )
{
   AuxInfo *aux = (AuxInfo *) ctx->DriverCtx;

   assert(aux->mContext == ctx );

   ctx->Driver.UpdateState = AuxInfo::UpdateState;
   ctx->Driver.SetDrawBuffer = AuxInfo::SetDrawBuffer;
   ctx->Driver.SetReadBuffer = AuxInfo::SetReadBuffer;
   ctx->Driver.Color = AuxInfo::Color;
   ctx->Driver.Index = AuxInfo::Index;
   ctx->Driver.ClearIndex = AuxInfo::ClearIndex;
   ctx->Driver.ClearColor = AuxInfo::ClearColor;
   ctx->Driver.GetBufferSize = AuxInfo::GetBufferSize;
   ctx->Driver.GetString = AuxInfo::GetString;
   ctx->Driver.Clear = AuxInfo::Clear;

   if (ctx->Color.DrawBuffer == GL_FRONT) {
      /* read/write front buffer */
      ctx->Driver.WriteRGBASpan = AuxInfo::WriteRGBASpanFront;
      ctx->Driver.WriteRGBSpan = AuxInfo::WriteRGBSpanFront;
      ctx->Driver.WriteRGBAPixels = AuxInfo::WriteRGBAPixelsFront;
      ctx->Driver.WriteMonoRGBASpan = AuxInfo::WriteMonoRGBASpanFront;
      ctx->Driver.WriteMonoRGBAPixels = AuxInfo::WriteMonoRGBAPixelsFront;
      ctx->Driver.WriteCI32Span = AuxInfo::WriteCI32SpanFront;
      ctx->Driver.WriteCI8Span = AuxInfo::WriteCI8SpanFront;
      ctx->Driver.WriteMonoCISpan = AuxInfo::WriteMonoCISpanFront;
      ctx->Driver.WriteCI32Pixels = AuxInfo::WriteCI32PixelsFront;
      ctx->Driver.WriteMonoCIPixels = AuxInfo::WriteMonoCIPixelsFront;
      ctx->Driver.ReadRGBASpan = AuxInfo::ReadRGBASpanFront;
      ctx->Driver.ReadRGBAPixels = AuxInfo::ReadRGBAPixelsFront;
      ctx->Driver.ReadCI32Span = AuxInfo::ReadCI32SpanFront;
      ctx->Driver.ReadCI32Pixels = AuxInfo::ReadCI32PixelsFront;
   }
   else {
      /* read/write back buffer */
      ctx->Driver.WriteRGBASpan = AuxInfo::WriteRGBASpanBack;
      ctx->Driver.WriteRGBSpan = AuxInfo::WriteRGBSpanBack;
      ctx->Driver.WriteRGBAPixels = AuxInfo::WriteRGBAPixelsBack;
      ctx->Driver.WriteMonoRGBASpan = AuxInfo::WriteMonoRGBASpanBack;
      ctx->Driver.WriteMonoRGBAPixels = AuxInfo::WriteMonoRGBAPixelsBack;
      ctx->Driver.WriteCI32Span = AuxInfo::WriteCI32SpanBack;
      ctx->Driver.WriteCI8Span = AuxInfo::WriteCI8SpanBack;
      ctx->Driver.WriteMonoCISpan = AuxInfo::WriteMonoCISpanBack;
      ctx->Driver.WriteCI32Pixels = AuxInfo::WriteCI32PixelsBack;
      ctx->Driver.WriteMonoCIPixels = AuxInfo::WriteMonoCIPixelsBack;
      ctx->Driver.ReadRGBASpan = AuxInfo::ReadRGBASpanBack;
      ctx->Driver.ReadRGBAPixels = AuxInfo::ReadRGBAPixelsBack;
      ctx->Driver.ReadCI32Span = AuxInfo::ReadCI32SpanBack;
      ctx->Driver.ReadCI32Pixels = AuxInfo::ReadCI32PixelsBack;
    }

	RivaUpdateState05(ctx);
}


void AuxInfo::ClearIndex(GLcontext *ctx, GLuint index)
{
   AuxInfo *aux = (AuxInfo *) ctx->DriverCtx;
   aux->mClearIndex = index;
}


void AuxInfo::ClearColor(GLcontext *ctx, GLubyte r, GLubyte g,
                         GLubyte b, GLubyte a)
{
   AuxInfo *aux = (AuxInfo *) ctx->DriverCtx;
   aux->mClearColor[BE_RCOMP] = r;
   aux->mClearColor[BE_GCOMP] = g;
   aux->mClearColor[BE_BCOMP] = b;
   aux->mClearColor[BE_ACOMP] = a;
   assert(aux->mBGLView);
}


GLbitfield AuxInfo::Clear(GLcontext *ctx, GLbitfield mask,
                               GLboolean all, GLint x, GLint y,
                               GLint width, GLint height)
{
   if (mask & DD_FRONT_LEFT_BIT)
	{
		LOG(2,("GLView: ClearFront called\n"));
		ClearFront(ctx, all, x, y, width, height);
		mask &= ~DD_FRONT_LEFT_BIT;
//rudolf:
		mask &= RivaClearBuffers05(ctx, mask, all, x, y, width, height);
	}
   if (mask & (DD_BACK_LEFT_BIT | DD_DEPTH_BIT))
	{
		LOG(2,("GLView: ClearBack called\n"));
	   AuxInfo *aux = (AuxInfo *) ctx->DriverCtx;

		/* rudolf: workaround for 3Dlife demo app,
		 * besides: can't hurt having this 'foolproof' check.
		 * (Be's lib has it as well one way or the other) */
		if (!aux->mBitmap)
		{
			LOG(8,("AuxInfo::ClearBack: WARNING, app should call glViewport() first!\n"));
			LOG(8,("AuxInfo::ClearBack: HACK, calling glViewport() for app...\n"));
			glViewport(0, 0, 
				(GLint)(FrontBuffer.right - FrontBuffer.left),
				(GLint)(FrontBuffer.bottom - FrontBuffer.top));
			LOG(8,("AuxInfo::ClearBack: HACK, done executing glViewport() for app!\n"));
		}

//rudolf:
	   if (0)//mask & DD_BACK_LEFT_BIT)
		{
			ClearBack(ctx, all, x, y, width, height);
			mask &= ~DD_BACK_LEFT_BIT;
		}
		mask &= RivaClearBuffers05(ctx, mask, all, x, y, width, height);
	}
	return mask;
}


void AuxInfo::ClearFront(GLcontext *ctx,
                         GLboolean all, GLint x, GLint y,
                         GLint width, GLint height)
{
   AuxInfo *aux = (AuxInfo *) ctx->DriverCtx;
   BGLView *bglview = aux->mBGLView;
   assert(bglview);

    RivaSync(0);

   bglview->SetHighColor(aux->mClearColor[BE_RCOMP],
                         aux->mClearColor[BE_GCOMP],
                         aux->mClearColor[BE_BCOMP],
                         aux->mClearColor[BE_ACOMP]);
   bglview->SetLowColor(aux->mClearColor[BE_RCOMP],
                        aux->mClearColor[BE_GCOMP],
                        aux->mClearColor[BE_BCOMP],
                        aux->mClearColor[BE_ACOMP]);
   if (all) {
      BRect b = bglview->Bounds();
      bglview->FillRect(b);
   }
   else {
      // XXX untested
      BRect b;
      b.left = x;
      b.right = x + width;
      b.bottom = aux->mHeight - y - 1;
      b.top = b.bottom - height;
      bglview->FillRect(b);
   }

   // restore drawing color
   bglview->SetHighColor(aux->mColor[BE_RCOMP],
                         aux->mColor[BE_GCOMP],
                         aux->mColor[BE_BCOMP],
                         aux->mColor[BE_ACOMP]);
   bglview->SetLowColor(aux->mColor[BE_RCOMP],
                        aux->mColor[BE_GCOMP],
                        aux->mColor[BE_BCOMP],
                        aux->mColor[BE_ACOMP]);
}


void AuxInfo::ClearBack(GLcontext *ctx,
                        GLboolean all, GLint x, GLint y,
                        GLint width, GLint height)
{
   AuxInfo *aux = (AuxInfo *) ctx->DriverCtx;

   BGLView *bglview = aux->mBGLView;
   assert(bglview);
   uint32 *start32 = (uint32 *) riva_sinfo[0].backbufmappedaddr;
   uint16 *start16 = (uint16 *) riva_sinfo[0].backbufmappedaddr; 
   const GLuint *clearPixelPtr = (const GLuint *) aux->mClearColor;
   const GLuint clearPixel = *clearPixelPtr;

   if (all)
   {
      const int numPixels = aux->mAlignedWidth * aux->mHeight;
      if (clearPixel == 0)
      {
		memset(start32, 0, numPixels * riva_sinfo[0].bytesperpixel);
      }
      else
      {
         for (int i = 0; i < numPixels; i++)
         {
			switch (si->dm.space)
			{
			case B_RGB32_LITTLE:
                start32[i] = clearPixel;
				break;
			case B_RGB16_LITTLE:
			case B_RGB15_LITTLE:
             	start16[i] = clearPixel;
				break;
			}
         }
      }
   }
   else
   {
      // XXX untested
      start32 += (y * aux->mAlignedWidth) + x;
      start16 += (y * aux->mAlignedWidth) + x;
      for (int i = 0; i < height; i++)
      {
         for (int j = 0; j < width; j++)
         {
			switch (si->dm.space)
			{
			case B_RGB32_LITTLE:
	            start32[j] = clearPixel;
				break;
			case B_RGB16_LITTLE:
			case B_RGB15_LITTLE:
	            start16[j] = clearPixel;
				break;
			}
         }
         start32 += aux->mAlignedWidth;
         start16 += aux->mAlignedWidth;
      }
   }
}


void AuxInfo::Index(GLcontext *ctx, GLuint index)
{
   AuxInfo *aux = (AuxInfo *) ctx->DriverCtx;
   BGLView *bglview = aux->mBGLView;
   assert(bglview);
   aux->mIndex = index;
}


void AuxInfo::Color(GLcontext *ctx, GLubyte r, GLubyte g,
						GLubyte b, GLubyte a)
{
   AuxInfo *aux = (AuxInfo *) ctx->DriverCtx;
   BGLView *bglview = aux->mBGLView;
   assert(bglview);
   aux->mColor[BE_RCOMP] = r;
   aux->mColor[BE_GCOMP] = g;
   aux->mColor[BE_BCOMP] = b;
   aux->mColor[BE_ACOMP] = a;
   bglview->SetHighColor(r, g, b, a);
   bglview->SetLowColor(r, g, b, a);
}

GLboolean AuxInfo::SetDrawBuffer(GLcontext *ctx, GLenum buffer)
{
	/* rudolf:
	 * actual access buffer is selected trough 'UpdateState' which selects functions
	 * that do the accesses. */

   if (buffer == GL_FRONT_LEFT)
      return GL_TRUE;
   else if (buffer == GL_BACK_LEFT)
      return GL_TRUE;
   else
      return GL_FALSE;
}

void AuxInfo::SetReadBuffer(GLcontext *ctx, GLframebuffer *colorBuffer,
                            GLenum buffer)
{
	/* rudolf:
	 * seems unneeded at this time, but I might be wrong..
	 * (access buffer is selected trough 'UpdateState' which selects functions
	 * that do the accesses.) */
}

void AuxInfo::GetBufferSize(GLcontext *ctx, GLuint *width,
                            GLuint *height)
{
	LOG(2,("AuxInfo::GetBufferSize called\n"));

   AuxInfo *aux = (AuxInfo *) ctx->DriverCtx;
   BGLView *bglview = aux->mBGLView;
   assert(bglview);
   BRect b = bglview->Bounds();

	LOG(2,("AuxInfo::GetBufferSize: current left-top is %f,%f; right-bottom is %f,%f\n",
		b.left, b.top, b.right, b.bottom));

	aux->mUnalignedWidth = *width = (GLuint) (b.right - b.left + 1);

	/* we have to account for engine granularity!!! (NV18: 64 bytes) */
	switch(si->dm.space)
	{
	case B_RGB15_LITTLE:
	case B_RGB16_LITTLE:
		*width = (*width + 0x1f) & ~0x1f;
		break;
	case B_RGB32_LITTLE:
		*width = (*width + 0x0f) & ~0x0f;
		break;
	default:
		*width = (*width + 0x3f) & ~0x3f;
		break;
	}

   *height = (GLuint) (b.bottom - b.top + 1);
   aux->mBottom = (GLint) b.bottom;

   if (ctx->Visual->DBflag)
   {
//rudolf: not correct enough, just skipping this check for now:
//      if (*width != aux->mAlignedWidth || *height != aux->mHeight)
      {
        // allocate new size of back buffer bitmap
		LOG(2,("backbuf adress is $%08x\n", (riva_sinfo[0].backbufmappedaddr)));

		/* set new buffer sizes */
		FrontBuffer.right = FrontBuffer.left + (aux->mUnalignedWidth - 1);
		FrontBuffer.bottom = FrontBuffer.top + (*height - 1);

		/* tell hardware to update it's buffers */
		riva_sinfo[0].SetSurfaces3D(
			((rivaContext.bufRender == GL_FRONT) ?
				riva_sinfo[0].frontbufferoffset :
				riva_sinfo[0].backbufferoffset),
			riva_sinfo[0].depthbufferoffset);

		aux->mBitmap = true;
      }
   }
   else
   {
      aux->mBitmap = false;
   }

	aux->mAlignedWidth = *width;
	aux->mHeight = *height;
}

//not used atm (RivaString is used instead)
const GLubyte *AuxInfo::GetString(GLcontext *ctx, GLenum name)
{
   switch (name) {
      case GL_RENDERER:
         return (const GLubyte *) "Mesa BeOS";
      default:
         // Let core library handle all other cases
         return NULL;
   }
}


// Plot a pixel.  (0,0) is upper-left corner
// This is only used when drawing to the front buffer.
static void Plot(BGLView *bglview, int x, int y)
{
   // XXX There's got to be a better way!
   BPoint p(x, y), q(x+1, y);
   bglview->StrokeLine(p, q);
}


void AuxInfo::WriteRGBASpanFront(const GLcontext *ctx, GLuint n,
                                 GLint x, GLint y,
                                 CONST GLubyte rgba[][4],
                                 const GLubyte mask[])
{
   AuxInfo *aux = (AuxInfo *) ctx->DriverCtx;
   BGLView *bglview = aux->mBGLView;
   assert(bglview);
   int flippedY = aux->mBottom - y;

   if (mask){
      for (GLuint i = 0; i < n; i++) {
         if (mask[i]) {
            bglview->SetHighColor(rgba[i][0], rgba[i][1], rgba[i][2], rgba[i][3]);
            Plot(bglview, x++, flippedY);
         }
      }
   }
   else {
      for (GLuint i = 0; i < n; i++) {
         bglview->SetHighColor(rgba[i][0], rgba[i][1], rgba[i][2], rgba[i][3]);
         Plot(bglview, x++, flippedY);
      }
   }
}

void AuxInfo::WriteRGBSpanFront(const GLcontext *ctx, GLuint n,
                                GLint x, GLint y,
                                CONST GLubyte rgba[][3],
                                const GLubyte mask[])
{
   AuxInfo *aux = (AuxInfo *) ctx->DriverCtx;
   BGLView *bglview = aux->mBGLView;
   assert(bglview);
   int flippedY = aux->mBottom - y;

   if (mask) {
      for (GLuint i = 0; i < n; i++) {
         if (mask[i]) {
            bglview->SetHighColor(rgba[i][0], rgba[i][1], rgba[i][2]);
            Plot(bglview, x++, flippedY);
         }
      }
   }
   else {
      for (GLuint i = 0; i < n; i++) {
         bglview->SetHighColor(rgba[i][0], rgba[i][1], rgba[i][2]);
         Plot(bglview, x++, flippedY);
      }
   }
}

void AuxInfo::WriteMonoRGBASpanFront(const GLcontext *ctx, GLuint n,
                                     GLint x, GLint y, const GLubyte mask[])
{
   AuxInfo *aux = (AuxInfo *) ctx->DriverCtx;
   BGLView *bglview = aux->mBGLView;
   assert(bglview);
   int flippedY = aux->mBottom - y;

   if (mask) {
      for (GLuint i = 0; i < n; i++) {
         if (mask[i]) {
            Plot(bglview, x++, flippedY);
         }
      }
   }
   else {
      for (GLuint i = 0; i < n; i++) {
         Plot(bglview, x++, flippedY);
      }
   }
}

void AuxInfo::WriteRGBAPixelsFront(const GLcontext *ctx,
                                   GLuint n, const GLint x[], const GLint y[],
                                   CONST GLubyte rgba[][4],
                                   const GLubyte mask[] )
{
   AuxInfo *aux = (AuxInfo *) ctx->DriverCtx;
   BGLView *bglview = aux->mBGLView;
   assert(bglview);

   if (mask) {
      for (GLuint i = 0; i < n; i++) {
         if (mask[i]) {
            bglview->SetHighColor(rgba[i][0], rgba[i][1], rgba[i][2]);
            Plot(bglview, x[i], aux->mBottom - y[i]);
         }
      }
   }
   else {
      for (GLuint i = 0; i < n; i++) {
         bglview->SetHighColor(rgba[i][0], rgba[i][1], rgba[i][2]);
         Plot(bglview, x[i], aux->mBottom - y[i]);
      }
   }
}

void AuxInfo::WriteMonoRGBAPixelsFront(const GLcontext *ctx, GLuint n,
                                       const GLint x[], const GLint y[],
                                       const GLubyte mask[])
{
   AuxInfo *aux = (AuxInfo *) ctx->DriverCtx;
   BGLView *bglview = aux->mBGLView;
   assert(bglview);

   // plot points using current color
   if (mask) {
      for (GLuint i = 0; i < n; i++) {
         if (mask[i]) {
            Plot(bglview, x[i], aux->mBottom - y[i]);
         }
      }
   }
   else {
      for (GLuint i = 0; i < n; i++) {
         Plot(bglview, x[i], aux->mBottom - y[i]);
      }
   }
}


void AuxInfo::WriteCI32SpanFront( const GLcontext *ctx, GLuint n, GLint x, GLint y,
                             const GLuint index[], const GLubyte mask[] )
{
   // XXX to do
}

void AuxInfo::WriteCI8SpanFront( const GLcontext *ctx, GLuint n, GLint x, GLint y,
                            const GLubyte index[], const GLubyte mask[] )
{
   // XXX to do
}

void AuxInfo::WriteMonoCISpanFront( const GLcontext *ctx, GLuint n,
                               GLint x, GLint y, const GLubyte mask[] )
{
   // XXX to do
}


void AuxInfo::WriteCI32PixelsFront( const GLcontext *ctx, GLuint n,
                                    const GLint x[], const GLint y[],
                                    const GLuint index[], const GLubyte mask[] )
{
   // XXX to do
}

void AuxInfo::WriteMonoCIPixelsFront( const GLcontext *ctx, GLuint n,
                                      const GLint x[], const GLint y[],
                                      const GLubyte mask[] )
{
   // XXX to do
}


void AuxInfo::ReadCI32SpanFront( const GLcontext *ctx,
                                 GLuint n, GLint x, GLint y, GLuint index[] )
{
   // XXX to do
}


void AuxInfo::ReadRGBASpanFront( const GLcontext *ctx, GLuint n,
                                 GLint x, GLint y, GLubyte rgba[][4] )
{
   // XXX to do
}


void AuxInfo::ReadCI32PixelsFront( const GLcontext *ctx,
                                   GLuint n, const GLint x[], const GLint y[],
                                   GLuint indx[], const GLubyte mask[] )
{
   // XXX to do
}


void AuxInfo::ReadRGBAPixelsFront( const GLcontext *ctx,
                                   GLuint n, const GLint x[], const GLint y[],
                                   GLubyte rgba[][4], const GLubyte mask[] )
{
   // XXX to do
}



void AuxInfo::WriteRGBASpanBack(const GLcontext *ctx, GLuint n,
                                 GLint x, GLint y,
                                 CONST GLubyte rgba[][4],
                                 const GLubyte mask[])
{
//rud this routine is used a lot:
   AuxInfo *aux = (AuxInfo *) ctx->DriverCtx;
   int row = aux->mBottom - y;
   GLubyte *pixel = (GLubyte *) riva_sinfo[0].backbufmappedaddr +
   	(row * aux->mAlignedWidth + x) * riva_sinfo[0].bytesperpixel;

	/* make sure hardware isn't drawing */
	if (hw_drawing) RivaSyncRender();

   if (mask)
   {
      for (GLuint i = 0; i < n; i++)
      {
         if (mask[i])
         {
            switch(si->dm.space)
            {
			case B_RGB32_LITTLE:
	            pixel[BE_RCOMP] = rgba[i][RCOMP];
    	        pixel[BE_GCOMP] = rgba[i][GCOMP];
            	pixel[BE_BCOMP] = rgba[i][BCOMP];
            	pixel[BE_ACOMP] = rgba[i][ACOMP];
		         break;
			case B_RGB16_LITTLE:
				pixel[0] = (rgba[i][BCOMP] >> 3) | ((rgba[i][GCOMP] & 0x1c) << 3);
				pixel[1] = (rgba[i][GCOMP] >> 5) | (rgba[i][RCOMP] & 0xf8);		
				break;
			case B_RGB15_LITTLE:
//rud: do..
				pixel[0] = (rgba[i][BCOMP] >> 3) | ((rgba[i][GCOMP] & 0x3f) << 2);
				pixel[1] = (rgba[i][GCOMP] >> 6) | ((rgba[i][RCOMP] & 0xf8) >> 1);		
				break;
			default:
//rud: do..
				break;
         	}
         }
         switch(si->dm.space)
         {
		case B_RGB32_LITTLE:
			pixel += 4;
			break;
		case B_RGB16_LITTLE:
		case B_RGB15_LITTLE:
			pixel += 2;
			break;
		default:
			pixel ++;
			break;
		}
      }
   }
   else {
      for (GLuint i = 0; i < n; i++) {
            switch(si->dm.space)
            {
			case B_RGB32_LITTLE:
	            pixel[BE_RCOMP] = rgba[i][RCOMP];
    	        pixel[BE_GCOMP] = rgba[i][GCOMP];
            	pixel[BE_BCOMP] = rgba[i][BCOMP];
            	pixel[BE_ACOMP] = rgba[i][ACOMP];
				pixel += 4;
		         break;
			case B_RGB16_LITTLE:
				pixel[0] = (rgba[i][BCOMP] >> 3) | ((rgba[i][GCOMP] & 0x1c) << 3);
				pixel[1] = (rgba[i][GCOMP] >> 5) | (rgba[i][RCOMP] & 0xf8);		
				pixel += 2;
				break;
			case B_RGB15_LITTLE:
//rud: do..
				pixel[0] = (rgba[i][BCOMP] >> 3) | ((rgba[i][GCOMP] & 0x3f) << 2);
				pixel[1] = (rgba[i][GCOMP] >> 6) | ((rgba[i][RCOMP] & 0xf8) >> 1);		
				pixel += 2;
				break;
			default:
//rud: do..
				pixel ++;
				break;
         	}
      }
   }
}


void AuxInfo::WriteRGBSpanBack(const GLcontext *ctx, GLuint n,
                                GLint x, GLint y,
                                CONST GLubyte rgb[][3],
                                const GLubyte mask[])
{
   AuxInfo *aux = (AuxInfo *) ctx->DriverCtx;
   int row = aux->mBottom - y;
   GLubyte *pixel = (GLubyte *) riva_sinfo[0].backbufmappedaddr +
    (row * aux->mAlignedWidth + x) * riva_sinfo[0].bytesperpixel;

	/* make sure hardware isn't drawing */
	if (hw_drawing) RivaSyncRender();

   if (mask) {
      for (GLuint i = 0; i < n; i++) {
         if (mask[i])
         {
            switch(si->dm.space)
            {
			case B_RGB32_LITTLE:
	            pixel[BE_RCOMP] = rgb[i][RCOMP];
    	        pixel[BE_GCOMP] = rgb[i][GCOMP];
            	pixel[BE_BCOMP] = rgb[i][BCOMP];
            	pixel[BE_ACOMP] = 0xff;
		         break;
			case B_RGB16_LITTLE:
				pixel[0] = (rgb[i][BCOMP] >> 3) | ((rgb[i][GCOMP] & 0x1c) << 3);
				pixel[1] = (rgb[i][GCOMP] >> 5) | (rgb[i][RCOMP] & 0xf8);		
				break;
			//rud: add rest..
         	}
         }
         switch(si->dm.space)
         {
		case B_RGB32_LITTLE:
			pixel += 4;
			break;
		case B_RGB16_LITTLE:
		case B_RGB15_LITTLE:
			pixel += 2;
			break;
		default:
			pixel ++;
			break;
		}
      }
   }
   else {
      for (GLuint i = 0; i < n; i++)
      {
            switch(si->dm.space)
            {
			case B_RGB32_LITTLE:
	            pixel[BE_RCOMP] = rgb[i][RCOMP];
    	        pixel[BE_GCOMP] = rgb[i][GCOMP];
            	pixel[BE_BCOMP] = rgb[i][BCOMP];
            	pixel[BE_ACOMP] = 0xff;
				pixel += 4;
		         break;
			case B_RGB16_LITTLE:
				pixel[0] = (rgb[i][BCOMP] >> 3) | ((rgb[i][GCOMP] & 0x1c) << 3);
				pixel[1] = (rgb[i][GCOMP] >> 5) | (rgb[i][RCOMP] & 0xf8);		
				pixel += 2;
				break;
			//rud: add rest..
         	}
      }
   }
}


//rud: from dd.h (mesa):
/* Write a horizontal run of RGBA pixels all with the color last
 * specified by the Color function.
 */
void AuxInfo::WriteMonoRGBASpanBack(const GLcontext *ctx, GLuint n,
                                     GLint x, GLint y, const GLubyte mask[])
{
   AuxInfo *aux = (AuxInfo *) ctx->DriverCtx;
   int row = aux->mBottom - y;

	/* make sure hardware isn't drawing */
	if (hw_drawing) RivaSyncRender();

   switch(si->dm.space)
   {
	case B_RGB32_LITTLE:
	{   
	   uint32* pixelPtr =
	   		(uint32*) riva_sinfo[0].backbufmappedaddr + (row * aux->mAlignedWidth) + x;
	   const uint32 pixel = *((uint32*) aux->mColor);
	   if (mask)
	   {
    	  for (GLuint i = 0; i < n; i++)
    	  {
        	 if (mask[i])
        	    *pixelPtr = pixel;
	         pixelPtr++;
	      }
	   }
	   else
	   {
    	  for (GLuint i = 0; i < n; i++)
    	     *pixelPtr++ = pixel;
	   }
		break;
	}
	case B_RGB16_LITTLE:
	{
	   uint16* pixelPtr =
	   		(uint16*) riva_sinfo[0].backbufmappedaddr + (row * aux->mAlignedWidth) + x;
		uint32 temp = *((uint32*) aux->mColor);
	   const uint16 pixel =
	   		((temp & 0x000000f8) >> 3) |
	   		((temp & 0x0000fc00) >> 5) |
	   		((temp & 0x00f80000) >> 8);
	   if (mask)
	   {
    	  for (GLuint i = 0; i < n; i++)
    	  {
        	 if (mask[i])
        	    *pixelPtr = pixel;
	         pixelPtr++;
	      }
	   }
	   else
	   {
    	  for (GLuint i = 0; i < n; i++)
	         *pixelPtr++ = pixel;
	   }
		break;
	}
//		rud: add rest..
	}
}


void AuxInfo::WriteRGBAPixelsBack(const GLcontext *ctx,
                                   GLuint n, const GLint x[], const GLint y[],
                                   CONST GLubyte rgba[][4],
                                   const GLubyte mask[] )
{
   AuxInfo *aux = (AuxInfo *) ctx->DriverCtx;

	/* make sure hardware isn't drawing */
	if (hw_drawing) RivaSyncRender();

   if (mask) {
      for (GLuint i = 0; i < n; i++) {
         if (mask[i]) {
            GLubyte *pixel = (GLubyte *) riva_sinfo[0].backbufmappedaddr +
            	(((aux->mBottom - y[i]) * aux->mAlignedWidth) + x[i]) * riva_sinfo[0].bytesperpixel;

            switch(si->dm.space)
            {
			case B_RGB32_LITTLE:
	            pixel[BE_RCOMP] = rgba[i][RCOMP];
    	        pixel[BE_GCOMP] = rgba[i][GCOMP];
            	pixel[BE_BCOMP] = rgba[i][BCOMP];
            	pixel[BE_ACOMP] = rgba[i][ACOMP];
		         break;
			case B_RGB16_LITTLE:
				pixel[0] = (rgba[i][BCOMP] >> 3) | ((rgba[i][GCOMP] & 0x1c) << 3);
				pixel[1] = (rgba[i][GCOMP] >> 5) | (rgba[i][RCOMP] & 0xf8);		
				break;
			case B_RGB15_LITTLE:
//rud: do..
				pixel[0] = (rgba[i][BCOMP] >> 3) | ((rgba[i][GCOMP] & 0x3f) << 2);
				pixel[1] = (rgba[i][GCOMP] >> 6) | ((rgba[i][RCOMP] & 0xf8) >> 1);		
				break;
			default:
//rud: do..
				break;
         	}
         }
      }
   }
   else {
      for (GLuint i = 0; i < n; i++)
      {
         	GLubyte *pixel = (GLubyte *) riva_sinfo[0].backbufmappedaddr +
            	(((aux->mBottom - y[i]) * aux->mAlignedWidth) + x[i]) * riva_sinfo[0].bytesperpixel;

            switch(si->dm.space)
            {
			case B_RGB32_LITTLE:
	            pixel[BE_RCOMP] = rgba[i][RCOMP];
    	        pixel[BE_GCOMP] = rgba[i][GCOMP];
            	pixel[BE_BCOMP] = rgba[i][BCOMP];
            	pixel[BE_ACOMP] = rgba[i][ACOMP];
		         break;
			case B_RGB16_LITTLE:
				pixel[0] = (rgba[i][BCOMP] >> 3) | ((rgba[i][GCOMP] & 0x1c) << 3);
				pixel[1] = (rgba[i][GCOMP] >> 5) | (rgba[i][RCOMP] & 0xf8);		
				break;
			case B_RGB15_LITTLE:
//rud: do..
				pixel[0] = (rgba[i][BCOMP] >> 3) | ((rgba[i][GCOMP] & 0x3f) << 2);
				pixel[1] = (rgba[i][GCOMP] >> 6) | ((rgba[i][RCOMP] & 0xf8) >> 1);		
				break;
			default:
//rud: do..
				break;
         	}
      }
   }
}


void AuxInfo::WriteMonoRGBAPixelsBack(const GLcontext *ctx, GLuint n,
                                       const GLint x[], const GLint y[],
                                       const GLubyte mask[])
{
   AuxInfo *aux = (AuxInfo *) ctx->DriverCtx;

	/* make sure hardware isn't drawing */
	if (hw_drawing) RivaSyncRender();

  switch(si->dm.space)
   {
	case B_RGB32_LITTLE:
	{   
	   const uint32 pixel = *((uint32*) aux->mColor);
	   if (mask)
	   {
    	  for (GLuint i = 0; i < n; i++)
    	  {
        	 if (mask[i])
        	 {
       	     	uint32* pixelPtr = (uint32*) riva_sinfo[0].backbufmappedaddr +
      	    		(aux->mBottom - y[i]) * aux->mAlignedWidth + x[i];
            	*pixelPtr = pixel;
	         }
    	  }
   		}
   		else
   		{
      		for (GLuint i = 0; i < n; i++)
      		{
      		   uint32* pixelPtr = (uint32*) riva_sinfo[0].backbufmappedaddr +
            	     (aux->mBottom - y[i]) * aux->mAlignedWidth + x[i];
         		*pixelPtr = pixel;
      		}
   		}
		break;
	}
	case B_RGB16_LITTLE:
	{   
		uint32 temp = *((uint32*) aux->mColor);
	   const uint16 pixel =
	   		((temp & 0x000000f8) >> 3) |
	   		((temp & 0x0000fc00) >> 5) |
	   		((temp & 0x00f80000) >> 8);
	   if (mask)
	   {
    	  for (GLuint i = 0; i < n; i++)
    	  {
        	 if (mask[i])
        	 {
       	     	uint16* pixelPtr = (uint16*) riva_sinfo[0].backbufmappedaddr +
        	    	        (aux->mBottom - y[i]) * aux->mAlignedWidth + x[i];
            	*pixelPtr = pixel;
	         }
    	  }
   		}
   		else
   		{
      		for (GLuint i = 0; i < n; i++)
      		{
      		   uint16* pixelPtr = (uint16*) riva_sinfo[0].backbufmappedaddr +
            	     (aux->mBottom - y[i]) * aux->mAlignedWidth + x[i];
         		*pixelPtr = pixel;
      		}
   		}
		break;
	}
//rud: add rest..
	}
}


void AuxInfo::WriteCI32SpanBack( const GLcontext *ctx, GLuint n,
                                 GLint x, GLint y,
                                 const GLuint index[], const GLubyte mask[] )
{
   // XXX to do
}

void AuxInfo::WriteCI8SpanBack( const GLcontext *ctx, GLuint n,
                                GLint x, GLint y,
                                const GLubyte index[], const GLubyte mask[] )
{
   // XXX to do
}

void AuxInfo::WriteMonoCISpanBack( const GLcontext *ctx, GLuint n,
                                   GLint x, GLint y, const GLubyte mask[] )
{
   // XXX to do
}


void AuxInfo::WriteCI32PixelsBack( const GLcontext *ctx, GLuint n,
                                   const GLint x[], const GLint y[],
                                   const GLuint index[], const GLubyte mask[] )
{
   // XXX to do
}

void AuxInfo::WriteMonoCIPixelsBack( const GLcontext *ctx, GLuint n,
                                     const GLint x[], const GLint y[],
                                     const GLubyte mask[] )
{
   // XXX to do
}


void AuxInfo::ReadCI32SpanBack( const GLcontext *ctx,
                                GLuint n, GLint x, GLint y, GLuint index[] )
{
   // XXX to do
}


void AuxInfo::ReadRGBASpanBack( const GLcontext *ctx, GLuint n,
                                GLint x, GLint y, GLubyte rgba[][4] )
{
// LOG(2,("ReadRGBASpanBack called!\n"));

   AuxInfo *aux = (AuxInfo *) ctx->DriverCtx;

   const GLubyte *pixel = (GLubyte *) riva_sinfo[0].backbufmappedaddr +
		(((aux->mBottom - y) * aux->mAlignedWidth) + x) * riva_sinfo[0].bytesperpixel;

	/* make sure hardware isn't drawing */
	if (hw_drawing) RivaSyncRender();

   for (GLuint i = 0; i < n; i++) {
            switch(si->dm.space)
            {
			case B_RGB32_LITTLE:
		    	rgba[i][ACOMP] = pixel[BE_ACOMP];
    			rgba[i][BCOMP] = pixel[BE_BCOMP];
		      	rgba[i][GCOMP] = pixel[BE_GCOMP];
		 	    rgba[i][RCOMP] = pixel[BE_RCOMP];
		         pixel += 4;
				break;
			case B_RGB16_LITTLE:
		    	rgba[i][ACOMP] = 0xff;
    			rgba[i][BCOMP] = (pixel[0] << 3);
		      	rgba[i][GCOMP] = (((pixel[0] & 0xe0) >> 3) | (pixel[1] << 5));
		 	    rgba[i][RCOMP] = (pixel[1] & 0xf8);
		         pixel += 2;
				break;
			//rud do rest..
         	}
   }
}


void AuxInfo::ReadCI32PixelsBack( const GLcontext *ctx,
                                   GLuint n, const GLint x[], const GLint y[],
                                   GLuint indx[], const GLubyte mask[] )
{
   // XXX to do
}


void AuxInfo::ReadRGBAPixelsBack( const GLcontext *ctx,
                                  GLuint n, const GLint x[], const GLint y[],
                                  GLubyte rgba[][4], const GLubyte mask[] )
{
   AuxInfo *aux = (AuxInfo *) ctx->DriverCtx;

	/* make sure hardware isn't drawing */
	if (hw_drawing) RivaSyncRender();

   for (GLuint i = 0; i < n; i++) {
      if (y[i] < aux->mHeight) {
         const GLubyte *pixel = (const GLubyte *) riva_sinfo[0].backbufmappedaddr +
            ((aux->mBottom - y[i]) * aux->mAlignedWidth + x[i]) *
            	riva_sinfo[0].bytesperpixel;

            switch(si->dm.space)
            {
			case B_RGB32_LITTLE:
		    	rgba[i][ACOMP] = pixel[BE_ACOMP];
    			rgba[i][BCOMP] = pixel[BE_BCOMP];
		      	rgba[i][GCOMP] = pixel[BE_GCOMP];
		 	    rgba[i][RCOMP] = pixel[BE_RCOMP];
				break;
			case B_RGB16_LITTLE:
		    	rgba[i][ACOMP] = 0xff;
    			rgba[i][BCOMP] = (pixel[0] << 3);
		      	rgba[i][GCOMP] = (((pixel[0] & 0xe0) >> 3) | (pixel[1] << 5));
		 	    rgba[i][RCOMP] = (pixel[1] & 0xf8);
				break;
			//rud do rest..
         	}
      }
   }
}




//------------------------------------------------------------------
// Public interface methods
//------------------------------------------------------------------


//
// Input:  rect - initial rectangle
//         name - window name
//         resizingMode - example: B_FOLLOW_NONE
//         mode - usually 0 ?
//         options - Bitwise-OR of BGL_* tokens
//
//rudolf: flags mentioned are _required_ material (for max Be compatibility)!
BGLView::BGLView(BRect rect, char *name,
                 ulong resizingMode, ulong mode,
                 ulong options)
   :BView(rect, name, resizingMode,
   	mode | B_WILL_DRAW | B_PULSE_NEEDED | B_FRAME_EVENTS)
{
	/* we don't support single buffering yet, force double buffering... */
	options |= BGL_DOUBLE;
	/* remember original options in case desktop colordepth changes.. */
	ogl_options = options;

	/* note:
	 * BGL_RGB means that options bit 0 is zero; BGL_INDEX has bit 0 high!
	 * so these BGLview colorspace related bit-defines are a bit error prone... (!) */
   const GLboolean rgbFlag = (options & BGL_INDEX) == BGL_RGB;
   const GLboolean alphaFlag = (options & BGL_ALPHA) == BGL_ALPHA;
   const GLboolean dblFlag = (options & BGL_DOUBLE) == BGL_DOUBLE;
   const GLboolean stereoFlag = false;
   const GLint depth = (options & BGL_DEPTH) ? 16 : 0;
   const GLint stencil = (options & BGL_STENCIL) ? 8 : 0;
   const GLint accum = (options & BGL_ACCUM) ? 16 : 0;
   const GLint index = (options & BGL_INDEX) ? 32 : 0;

   /* note:
    * we should set the depth according to the really used depth for the colorbuffers.
    * Although our colorbuffer accessroutines translate all calls between the
    * current buffer's space and 32-bit depth for the caller, specifying the actually
    * used space here sets up Mesa's internal value-rounding features: which is
    * 'correct'.
    *
    * Faults here are 'easy' testable for instance with the GLteapot: grabbing it
    * doesn't work probably if the space is set wrong (pixels are readback by the app:
    * this (should) invoke(s) our 'ReadRGBASpanBack' function. */
   
   /* first startup the driver... */
   if (init_3da() != B_OK)
   {
      fprintf(stderr, "3DA Warning: cloning failed!\n");
   }

/*
	//old stuff: backbuffer was always 32bit
   const GLint red = rgbFlag ? 8 : 0;
   const GLint green = rgbFlag ? 8 : 0;
   const GLint blue = rgbFlag ? 8 : 0;
   const GLint alpha = alphaFlag ? 8 : 0;
*/

	/* ... now setup the info for the colorspace we will work in */
	GLint red = 0;
	GLint green = 0;
	GLint blue = 0;
	GLint alpha = 0;

	switch (si->dm.space)
	{
	case B_RGB32_LITTLE:
	case B_RGBA32_LITTLE:
		red = rgbFlag ? 8 : 0;
		green = rgbFlag ? 8 : 0;
		blue = rgbFlag ? 8 : 0;
		//fixme?
		alpha = alphaFlag ? 8 : 0;
		break;
	case B_RGB16_LITTLE:
		red = rgbFlag ? 5 : 0;
		green = rgbFlag ? 6 : 0;
		blue = rgbFlag ? 5 : 0;
		alpha = 0;
		break;
	case B_RGB15_LITTLE:
		red = rgbFlag ? 5 : 0;
		green = rgbFlag ? 5 : 0;
		blue = rgbFlag ? 5 : 0;
		//fixme?
		alpha = alphaFlag ? 1 : 0;
		break;
	default:
		//fixme: checkout indexed mode B_CMAP8
		break;
	}

   if (!rgbFlag)
   {
      fprintf(stderr, "Mesa Warning: color index mode not supported\n");
   }


   // Allocate auxiliary data object
   AuxInfo *aux = new AuxInfo;

	drawing_lock = create_sem(0, "BGLView draw locker");

//rud: _create_visual can do this:
//   if (depthBits > (GLint) (8*sizeof(GLdepth))) {
      /* can't meet depth buffer requirements */
  //    return NULL;
//

   // examine option flags and create gl_context struct
   GLvisual *visual = _mesa_create_visual(rgbFlag, dblFlag, stereoFlag,
                                          red, green, blue, alpha,
										  index, depth, stencil,                                        
                                          accum, accum, accum, accum, 0);

   // create core framebuffer
   GLframebuffer *buffer = gl_create_framebuffer(visual,
                                              depth > 0 ? GL_TRUE : GL_FALSE,
//                                              GL_FALSE,//no software depth
                                              stencil > 0 ? GL_TRUE: GL_FALSE,
                                              accum > 0 ? GL_TRUE : GL_FALSE,
                                              alphaFlag
                                              );

//	buffer->UseSoftwareDepthBuffer = FALSE;
   // create core context
   const GLboolean direct = GL_TRUE;
   GLcontext *ctx = gl_create_context( visual, NULL, aux, direct );

   aux->Init(this, ctx, visual, buffer );

   // Hook aux data into BGLView object
   m_gc = aux;

//nv3d
	FrontBuffer = rect;

	void *handle = NULL;
   if (!nvHookServerSymbols(handle))
      fprintf(stderr, "3DA Warning: nVidia init went wrong!\n");

	LOG(2,("BGLView constructor: left-top is %f,%f;\n",
		FrontBuffer.left, FrontBuffer.top));
	LOG(2,("BGLView constructor: right-bottom is %f,%f\n",
		FrontBuffer.right, FrontBuffer.bottom));

//rud: quake2 fix from mesa 6.2:
	if (!gl_get_current_context())
	{
		LOG(8,("BGLView constructor: calling LockGL().\n"));
		LockGL();
	}

	//we'll use a hardware Z-buffer
	ctx->DrawBuffer->UseSoftwareDepthBuffer = FALSE;
	RivaCreateDepthBuffer (ctx);
}


BGLView::~BGLView()
{
   printf("BGLView destructor\n");

   if (!nvUnhookServerSymbols()) {
      fprintf(stderr, "3DA Warning: nVidia uninit went wrong!\n");
   }

   AuxInfo *aux = (AuxInfo *) m_gc;
   assert(aux);
   delete aux;

//rudolf:
	uninit_3da();

	/* delete the direct screen info (if it exists) */
	if (fClipList)
	{
		free(fClipList);
		fClipList = NULL;
	}

	delete_sem(drawing_lock);
}

void BGLView::LockGL()
{
   AuxInfo *aux = (AuxInfo *) m_gc;
   assert(aux);
   aux->MakeCurrent();

   /* Locking the looper: Rendering may never be interrupted by system events such
    * as keyboard/mouse input, window resize events etc.
    * Note:
    * Lock the looper as last thing in this routine, it must be locked as short as
    * can be! (to minimize lagging of window events, keyboard handling lag quake 1) */
   LockLooper();
}

void BGLView::UnlockGL()
{
   AuxInfo *aux = (AuxInfo *) m_gc;
   assert(aux);

   if (Looper()->IsLocked()) UnlockLooper();

   // Could call gl_make_current(NULL, NULL) but it would just
   // hinder performance

  /* stimulate system thread context switches so all pending events are handled!! */
//not practical: slows rendering down on single-CPU systems!
//  for (uint8 a = 0; a < 10; a++) snooze(1);
}

void BGLView::SwapBuffers()
{
   AuxInfo *aux = (AuxInfo *) m_gc;
   assert(aux);
   LockLooper();
   aux->SwapBuffers(true);
   UnlockLooper();
}

void BGLView::CopySubBufferMESA(GLint x, GLint y, GLuint width, GLuint height)
{
   AuxInfo *aux = (AuxInfo *) m_gc;
   assert(aux);
   LockLooper();
   aux->CopySubBuffer(x, y, width, height);
   UnlockLooper();
}


BView *BGLView::EmbeddedView()
{
   // XXX to do

	/* see bebook: */
	return NULL;
}

status_t BGLView::CopyPixelsOut(BPoint source, BBitmap *dest)
{
   // XXX to do
	return B_UNSUPPORTED;
}


status_t BGLView::CopyPixelsIn(BBitmap *source, BPoint dest)
{
   // XXX to do
	return B_UNSUPPORTED;
}

void BGLView::ErrorCallback(unsigned long errorCode)
{
	char msg[32];
	sprintf(msg, "GL: Error code $%04lx.", errorCode);
	// debugger(msg);
	printf("%s\n", msg);
}

/* note: 
 * we simply assume the backbuffer is still valid, and use that as source (Mesa 6.2.1).
 */
void BGLView::Draw(BRect updateRect)
{
   AuxInfo *aux = (AuxInfo *) m_gc;
   assert(aux);
   LockLooper();
   aux->SwapBuffers(false);
   UnlockLooper();
}

void BGLView::AttachedToWindow()
{
   BView::AttachedToWindow();

   // don't paint window background white when resized
   SetViewColor(B_TRANSPARENT_32_BIT);
}

void BGLView::AllAttached()
{
   BView::AllAttached();
//   printf("BGLView AllAttached\n");
}

void BGLView::DetachedFromWindow()
{
   BView::DetachedFromWindow();
}

void BGLView::AllDetached()
{
   BView::AllDetached();
//   printf("BGLView AllDetached");
}

void BGLView::FrameResized(float width, float height)
{
   AuxInfo *aux = (AuxInfo *) m_gc;
   assert(aux);
	/* yes, we have another bug in BeOS (both R5 and Dano)!! 
	 * the given width and height are the PREVIOUS one instead of the newest one.
	 * Get the newest rect ourself! (workaround found in Be's Teapot code.) */   
   BRect b = Bounds();
   width = b.Width() + 1;
   height = b.Height() + 1;
   aux->Resize((GLuint)width, (GLuint)height);

   BView::FrameResized(width, height);
}

status_t BGLView::Perform(perform_code d, void *arg)
{
   return BView::Perform(d, arg);
}


status_t BGLView::Archive(BMessage *data, bool deep) const
{
   return BView::Archive(data, deep);
}

void BGLView::MessageReceived(BMessage *msg)
{
   BView::MessageReceived(msg);
}

void BGLView::SetResizingMode(uint32 mode)
{
   BView::SetResizingMode(mode);
}

void BGLView::Show()
{
//   printf("BGLView Show\n");

	/* make a explicit full hide/show cycle for the Window to keep Quake2 rendering
	 * in direct mode! (fix for R5 left-top video position error on 'vid_restart') */
	LockLooper();
	Window()->Hide();
	Window()->Show();
	UnlockLooper();

	BView::Show();
}

void BGLView::Hide()
{
//   printf("BGLView Hide\n");
   BView::Hide();
}

BHandler *BGLView::ResolveSpecifier(BMessage *msg, int32 index,
                                    BMessage *specifier, int32 form,
                                    const char *property)
{
   return BView::ResolveSpecifier(msg, index, specifier, form, property);
}

status_t BGLView::GetSupportedSuites(BMessage *data)
{
   return BView::GetSupportedSuites(data);
}

void BGLView::DirectConnected( direct_buffer_info *info )
{
	LOG(2,("BGLView::DirectConnected: called\n"));

	/* you need to use that mask to read the buffer state */
	switch (info->buffer_state & B_DIRECT_MODE_MASK) {
	/* start a direct screen connection */
	case B_DIRECT_START :
		/* note that we need to reset the driver */
		if (info->driver_state == B_MODE_CHANGED)
		{
			//printf(">>>MODE_CHANGED<<<\n");
			driver_updated = false;
		}
		/* update the direct screen infos */
		if (fClipList)
		{
			free(fClipList);
			fClipList = NULL;
		}
		fWindowBounds = info->window_bounds;
		fClipBounds = info->clip_bounds;
		fNumClipRects = info->clip_list_count;
		/* fetch and correct the cliplist (menubar workaround) */
		fClipList = (clipping_rect *) malloc(fNumClipRects * sizeof(clipping_rect));
		if (fClipList)
		{
			memcpy(fClipList, info->clip_list, fNumClipRects * sizeof(clipping_rect));
			LOG(4,("B_DIRECT_START: # cliprects before fix %d\n", fNumClipRects));
			for (int32 cnt = 0; cnt < fNumClipRects; cnt++)
			{
				LOG(4,("cliplist[%d] before fix %d,%d;%d,%d\n", cnt,
					fClipList[cnt].left, fClipList[cnt].top,
					fClipList[cnt].right, fClipList[cnt].bottom));
			}
			clipping_rect * src = fClipList;
			clipping_rect * dst = &fClipList[fNumClipRects];
			for (uint32 cnt = 0; cnt < info->clip_list_count; cnt++)
			{
				if (src->top < (fWindowBounds.top + menu_offset))
					src->top = (fWindowBounds.top + menu_offset);
				if (src->top <= src->bottom)
				{
					src++;
				}
				else
				{
					*dst--;
					*src = *dst;
					fNumClipRects--;
				}
			}
			LOG(4,("B_DIRECT_START: # cliprects after fix %d\n", fNumClipRects));
			for (int32 cnt = 0; cnt < fNumClipRects; cnt++)
			{
				LOG(4,("cliplist[%d] after fix %d,%d;%d,%d\n", cnt,
					fClipList[cnt].left, fClipList[cnt].top,
					fClipList[cnt].right, fClipList[cnt].bottom));
			}
		}
		/* unblock the rendering functions */
		release_sem(drawing_lock);
		break;
	/* stop a direct screen connection */
	case B_DIRECT_STOP :
		/* system fixme?:
		 * would love to do this (below), but it's undefined here according to BeBook!
		 * We'd need this to be true _before_ the actual mode change is done, _and_
		 * _both_ for workspace changes AND mode changes (all updates of HW mode!!!)
		 *
		 * Current 'workaround': There's a special flag in the 2D driver indicating
		 * all relevant changes instead (si->mode_changed). */
/*
		if (info->driver_state == B_MODE_CHANGED)
		{
			printf(">>>B_DIRECT_STOP: MODE_CHANGED<<<\n");
		}
*/
		/* block the rendering functions */
		/* HACK alert: need timeout to get quake2 going (game modeswitches) !! */
		acquire_sem_etc(drawing_lock, 1, B_RELATIVE_TIMEOUT, 1000000);
		break;
	/* modify the state of a direct screen connection */
	case B_DIRECT_MODIFY :
		/* block the rendering functions */
		/* HACK alert: need timeout to get quake2 going (game startup) !! */
		acquire_sem_etc(drawing_lock, 1, B_RELATIVE_TIMEOUT, 1000000);
		/* note that we need to reset the driver */
		if (info->driver_state == B_MODE_CHANGED)
		{
			//printf(">>>MODE_CHANGED<<<\n");
			driver_updated = false;
		}
		/* update the direct screen infos */
		if (fClipList)
		{
			free(fClipList);
			fClipList = NULL;
		}
		fWindowBounds = info->window_bounds;
		fClipBounds = info->clip_bounds;
		fNumClipRects = info->clip_list_count;
		/* fetch and correct the cliplist (menubar workaround) */
		fClipList = (clipping_rect *) malloc(fNumClipRects * sizeof(clipping_rect));
		if (fClipList)
		{
			memcpy(fClipList, info->clip_list, fNumClipRects * sizeof(clipping_rect));
			LOG(4,("B_DIRECT_MODIFY: # cliprects before fix %d\n", fNumClipRects));
			for (int32 cnt = 0; cnt < fNumClipRects; cnt++)
			{
				LOG(4,("cliplist[%d] before fix %d,%d;%d,%d\n", cnt,
					fClipList[cnt].left, fClipList[cnt].top,
					fClipList[cnt].right, fClipList[cnt].bottom));
			}
			clipping_rect * src = fClipList;
			clipping_rect * dst = &fClipList[fNumClipRects];
			for (uint32 cnt = 0; cnt < info->clip_list_count; cnt++)
			{
				if (src->top < (fWindowBounds.top + menu_offset))
					src->top = (fWindowBounds.top + menu_offset);
				if (src->top <= src->bottom)
				{
					src++;
				}
				else
				{
					*dst--;
					*src = *dst;
					fNumClipRects--;
				}
			}
			LOG(4,("B_DIRECT_MODIFY: # cliprects after fix %d\n", fNumClipRects));
			for (int32 cnt = 0; cnt < fNumClipRects; cnt++)
			{
				LOG(4,("cliplist[%d] after fix %d,%d;%d,%d\n", cnt,
					fClipList[cnt].left, fClipList[cnt].top,
					fClipList[cnt].right, fClipList[cnt].bottom));
			}
		}
		/* unblock the rendering functions */
		release_sem(drawing_lock);
		break;
	default :
		break;
	}
}

/* This function should only be called _once_, upon app startup, to tell the
 * driver that it is working with a BDirectWindow by specifying TRUE as argument.
 * When an app is working with a BWindow this function should _not_ be called. */
void BGLView::EnableDirectMode( bool enabled )
{
	/* we enforce correct use of the EnableDirectMode function as some apps
	 * 'misuse' it (GLteapot disables DirectMode at app shutdown). */
	if (enabled && !DirectMode)
	{
		DirectMode = true;
		/* As some apps call SwapBuffers() before setting DirectMode (GLteapot),
		 * we need to instruct SwapBuffers() to recalculate the menubar offset
		 * (cliplist workaround code):
		 * We might have been given a clipping_list via DirectConnected() already
		 * that needs fixing! */
		menu_offset_done = false;
		LOG(2,("BGLView::EnableDirectMode: enabled DirectMode\n"));
	}
	else
	{
		/* if an app is going to switch back from a BDirectWindow to a BWindow,
		 * the BGLView instance has to be destroyed and a new instance should be
		 * created. Doing this will let the driver work in non-direct mode then. */
		if (DirectMode)
			LOG(2,("BGLView::EnableDirectMode: not touching DirectMode: kept ON\n"));
		else
			LOG(2,("BGLView::EnableDirectMode: not touching DirectMode: kept OFF\n"));
	}
}



//---- private methods ----------

void BGLView::_ReservedGLView1() {}
void BGLView::_ReservedGLView2() {}
void BGLView::_ReservedGLView3() {}
void BGLView::_ReservedGLView4() {}
void BGLView::_ReservedGLView5() {}
void BGLView::_ReservedGLView6() {}
void BGLView::_ReservedGLView7() {}
void BGLView::_ReservedGLView8() {}

#if 0
BGLView::BGLView(const BGLView &v)
	: BView(v)
{
   // XXX not sure how this should work
   printf("Warning BGLView::copy constructor not implemented\n");
}
#endif


BGLView &BGLView::operator=(const BGLView &v)
{
   printf("Warning BGLView::operator= not implemented\n");
	return *this;
}

void BGLView::dither_front()
{
   // no-op
}

bool BGLView::confirm_dither()
{
   // no-op
   return false;
}

void BGLView::draw(BRect r)
{
   // XXX no-op ???
}

/* Direct Window stuff */
void BGLView::drawScanline( int x1, int x2, int y, void *data )
{
   // no-op
}

void BGLView::scanlineHandler(struct rasStateRec *state,
                              GLint x1, GLint x2)
{
   // no-op
}

void BGLView::lock_draw()
{
   // no-op
}

void BGLView::unlock_draw()
{
   // no-op
}

bool BGLView::validateView()
{
   // no-op
   return true;
}

